<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLTranslation\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormFactoryInterface;
use Joomla\CMS\MVC\Model\ItemModel;
use Joomla\Component\JLTranslation\Administrator\Table\TranslationTable;
use Joomla\Database\ParameterType;
use Joomla\Plugin\System\JLTranslation\Extension\TranslationExtensionAbstract;
use Joomla\Plugin\System\JLTranslation\Helper\TranslationHelper;
use Joomla\Plugin\System\Joomlab\Helper\LanguageHelper;
use RuntimeException;
use SimpleXMLElement;

class TranslationModel extends ItemModel
{
    protected string $context = 'com_jltranslation.translation';

    public function getForms(): array
    {
        $formFactory     = Factory::getContainer()->get(FormFactoryInterface::class);
        $class           = $this->getExtensionClass();
        $classForm       = $class->prepareTranslationForm();
        $originalForm    = $formFactory->createForm($this->context, ['control' => 'jform']);
        $translationForm = $formFactory->createForm($this->context, ['control' => 'JLTranslation']);
        $formXml         = new SimpleXMLElement('<form><config><inlinehelp button="show"/></config></form>');
        $general         = $formXml->addChild('fieldset');
        $general->addAttribute('name', 'general');
        $groups = [];

        foreach ($class->getTranslationFields() as $fieldName) {
            $group = null;
            $name  = $fieldName;

            if (str_contains($name, '.')) {
                [$group, $name] = explode('.', $name, 2);
            }

            if (!($fieldXml = $classForm->getFieldXml($name, $group))) {
                continue;
            }

            if ($group) {
                if (!isset($groups[$group])) {
                    $groups[$group] = $formXml->addChild('fields');
                    $groups[$group]->addAttribute('name', $group);
                }

                $fieldSet = $groups[$group]->addChild('fieldset');
                $fieldSet->addAttribute('name', $group);
                $fieldSet->addAttribute('label', 'COM_JLTRANSLATION_GROUP_' . strtoupper($group) . '_LABEL');
                $field = $fieldSet->addChild('field');
            } else {
                $field = $general->addChild('field');
            }

            $this->appendXMLChildren($fieldXml, $field);
        }

        // Load com_fields
        if ($fieldSets = $classForm->getFieldsets('com_fields')) {
            $config            = ComponentHelper::getParams('com_jltranslation');
            $excludeFieldNames = preg_split('/\s*,\s*/', $config->get('excludeFieldNames', ''), -1, PREG_SPLIT_NO_EMPTY);
            $validFieldTypes   = preg_split('/\s*,\s*/', $config->get('validFieldTypes', 'calendar,color,editor,integer,imagelist,accessiblemedia,text,textarea,url'), -1, PREG_SPLIT_NO_EMPTY);
            $fields            = $formXml->addChild('fields');
            $fields->addAttribute('name', 'com_fields');

            foreach ($classForm->getFieldsets('com_fields') as $fieldset) {
                $set = $fields->addChild('fieldset');
                $set->addAttribute('name', $fieldset->name);
                $set->addAttribute('label', $fieldset->label);
                $count = 0;

                foreach ($classForm->getFieldset($fieldset->name) as $field) {
                    if (in_array(strtolower($field->type), $validFieldTypes)
                        && !in_array($field->fieldname, $excludeFieldNames)
                        && ($fieldXml = $classForm->getFieldXml($field->fieldname, 'com_fields'))
                    ) {
                        $count++;
                        $field = $set->addChild('field');
                        $this->appendXMLChildren($fieldXml, $field);
                    }
                }

                if (!$count) {
                    $dom = dom_import_simplexml($set);
                    $dom->parentNode->removeChild($dom);
                }
            }
        }


        $item = $this->getItem();

        // Load original form
        $originalForm->load($formXml);
        $originalForm->bind($classForm->getData()->toArray() ?: (array)$item);

        // Load translation form
        $translationForm->load(clone $formXml);
        $language        = $this->getState('filter.language');
        $context         = $this->getExtensionClass()->getBaseContext() . '.' . $item->{$class->getKey()};
        $translationData = TranslationHelper::getTranslationData($context, $language);
        $this->setState('translation.enabled', TranslationHelper::isEnabled($context, $language));
        $translationForm->bind($translationData);

        return [$originalForm, $translationForm];
    }

    public function getExtensionClass(): TranslationExtensionAbstract
    {
        static $class = null;

        if ($class instanceof TranslationExtensionAbstract) {
            return $class;
        }

        $filter = Factory::getApplication()->getInput()->get('filter', [], 'ARRAY');

        if (!($table = $filter['table'] ?? null) || !($class = TranslationHelper::getClassByTable($table))) {
            throw new RuntimeException('Invalid reference table');
        }

        $language = $filter['language'] ?? null;

        if (!in_array($language, LanguageHelper::getLanguageCodes())) {
            throw new RuntimeException('Invalid language code');
        }

        $this->setState('filter.table', $table);
        $this->setState('filter.language', $language);

        return $class;
    }

    protected function appendXMLChildren(SimpleXMLElement $fieldOrigin, SimpleXMLElement $fieldTarget)
    {
        foreach ($fieldOrigin->attributes() as $name => $value) {
            if ($name !== 'required') {
                $fieldTarget->addAttribute($name, $value);
            }
        }

        foreach ($fieldOrigin->children() as $child) {
            $childField = $fieldTarget->addChild($child->getName(), (string)$child);

            foreach ($child->attributes() as $name => $value) {
                $childField->addAttribute($name, $value);
            }

            if ($child->hasChildren()) {
                foreach ($child->children() as $c) {
                    $this->appendXMLChildren($childField, $c);
                }
            }
        }
    }

    public function getItem($pk = null)
    {
        if (!is_array($this->_item)) {
            $this->_item = [];
        }

        $pk = $pk ?: Factory::getApplication()->getInput()->getUint('id');

        if (!isset($this->_item[$pk])) {
            $class = $this->getExtensionClass();
            $db    = $this->getDatabase();
            $query = $db->getQuery(true);
            $idKey = $class->getKey();
            $query->select('a.*')
                ->from($db->quoteName('#__' . $class->getReferenceTable(), 'a'))
                ->where($db->quoteName('a.' . $idKey) . ' = :id')
                ->bind(':id', $pk, ParameterType::INTEGER);

            if (($this->_item[$pk] = $db->setQuery($query)->loadObject()) && empty($this->_item[$pk]->id)) {
                $this->_item[$pk]->id = $this->_item[$pk]->{$idKey};
            }
        }

        return $this->_item[$pk];
    }

    public function save(array $data): bool
    {
        /** @var TranslationTable $table */
        $class    = $this->getExtensionClass();
        $item     = $this->getItem();
        $language = $this->getState('filter.language');
        $context  = $class->getBaseContext() . '.' . $item->{$class->getKey()};

        $class->prepareSaveData($context, $language, $item, $data);
        $table    = $this->getTable('Translation', 'Administrator');
        $bindData = [
            'state'    => Factory::getApplication()->getInput()->getUint('state', 0) ? 1 : 0,
            'context'  => $context,
            'language' => $language,
            'data'     => $data,
        ];

        // Try to load the current data first
        $table->load(['context' => $context, 'language' => $language]);

        if (!$table->bind($bindData) || !$table->store()) {
            throw new RuntimeException('Unable to save translation');
        }

        return true;
    }
}