<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_jltranslation
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\JLTranslation\Driver;

defined('_JEXEC') or die;

use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\DatabaseQuery;
use Joomla\Database\Exception\PrepareStatementFailureException;
use Joomla\Database\FetchOrientation;
use Joomla\Database\Mysql\MysqlDriver;
use Joomla\Database\Mysqli\MysqliDriver;
use Joomla\Database\Mysqli\MysqliStatement;
use Joomla\Database\Pdo\PdoStatement;
use Joomla\Database\StatementInterface;
use Joomla\Plugin\System\JLTranslation\Extension\TranslationExtensionAbstract;
use Joomla\Plugin\System\JLTranslation\Helper\TranslationHelper;
use PhpMyAdmin\SqlParser\Components\Expression;
use PhpMyAdmin\SqlParser\Components\JoinKeyword;
use PhpMyAdmin\SqlParser\Parser;
use stdClass;
use Throwable;

$database = Factory::getContainer()->get(DatabaseInterface::class);

if ($database instanceof MysqlDriver) {
    class BaseStatement extends PdoStatement {}

    class BaseDatabaseDriver extends MysqlDriver {}
} elseif ($database instanceof MysqliDriver) {
    class BaseStatement extends MysqliStatement {}

    class BaseDatabaseDriver extends MysqliDriver {}
}

if (class_exists('Joomla\\Plugin\\System\\JLTranslation\\Driver\\BaseDatabaseDriver')) {
    require_once JPATH_PLUGINS . '/system/jltranslation/vendor/autoload.php';

    class DatabaseStatement extends BaseStatement
    {
        protected array $tablesMeta = [];

        public function __construct(array $options)
        {
            $this->tablesMeta = $options['tablesMeta'];
            parent::__construct($options['statement']);
        }

        public function fetch(?int $fetchStyle = null, int $cursorOrientation = FetchOrientation::NEXT, int $cursorOffset = 0)
        {
            $result      = parent::fetch($fetchStyle, $cursorOrientation, $cursorOffset);
            $validResult = is_object($result) || is_array($result);

            if ($validResult) {
                TranslationHelper::prepareStatementFetchResult($result);
            }

            // Translate with tables meta
            if ($this->tablesMeta && $validResult) {
                $hasProperty = function ($prop) use ($result) {
                    return is_object($result) ? property_exists($result, $prop) : array_key_exists($prop, $result);
                };

                $getProperty = function ($prop) use ($result) {
                    return is_object($result) ? $result->{$prop} : $result[$prop];
                };

                $setProperty = function ($prop, $val) use (&$result) {
                    if (is_object($result)) {
                        $result->{$prop} = $val;
                    } else {
                        $result[$prop] = $val;
                    }
                };

                foreach ($this->tablesMeta as $tableMeta) {
                    $table  = $tableMeta['table'];
                    $fields = $tableMeta['fields'];

                    // Collect item data
                    $item  = new stdClass();
                    $idKey = TranslationHelper::getClassByTable($table)->getKey();

                    // Attach item ID
                    if ($hasProperty($idKey)) {
                        $item->{$idKey} = $getProperty($idKey);
                    }

                    foreach ($fields as $field) {
                        [$alias, $column] = $field;
                        $property = $alias ?: $column;

                        if ($hasProperty($property)) {
                            $item->{$column} = $getProperty($property);
                        }
                    }

                    // Translate data
                    TranslationHelper::translate($table, $item);

                    foreach ($fields as $field) {
                        [$alias, $column] = $field;
                        $property = $alias ?: $column;

                        if ($hasProperty($property) && $getProperty($property) != $item->{$column}) {
                            $setProperty($property, $item->{$column});
                        }
                    }
                }
            }

            return $result;
        }
    }

    class DatabaseDriver extends BaseDatabaseDriver
    {
        protected array $tablesMeta = [];

        public function setQuery($query, $offset = 0, $limit = 0)
        {
            $boundedData = ($query instanceof DatabaseQuery) ? $query->getBounded() : [];
            $defaultLang = ComponentHelper::getParams('com_languages')->get('site', 'en-GB');
            $currentLang = Factory::getApplication()->getLanguage()->getTag();

            // Parse table meta
            $tablesMeta  = [];
            $tablePrefix = $this->tablePrefix;
            $isHika = 'SELECT v.*, c.*  FROM #__hikashop_variant AS v  INNER JOIN #__hikashop_characteristic AS c ON v.variant_characteristic_id = c.characteristic_id  WHERE v.variant_product_id IN (1,2,3)  ORDER BY v.ordering ASC, c.characteristic_ordering ASC' == trim((string) $query);

            $sql         = str_replace('#__', $tablePrefix, trim((string) $query));

            if (str_starts_with(strtoupper($sql), 'SELECT ')
                && !str_starts_with(strtoupper($sql), 'SELECT COUNT')
                && $defaultLang !== $currentLang
            ) {
                try {
                    $parser    = new Parser($sql);
                    $statement = $parser->statements[0];

                    if (!empty($statement->expr)) {
                        $tablesMap = [];
                        $mapTable  = function (Expression $expr) use (&$tablesMap, $tablePrefix) {
                            $alias = str_replace($tablePrefix, '', ($expr->alias ?: $expr->table) ?? '');

                            if (is_string($alias) && is_string($expr->table)) {
                                $table             = str_replace($tablePrefix, '', $alias);
                                $tablesMap[$table] = str_replace($tablePrefix, '', $expr->table);
                            }
                        };

                        if (!empty($statement->from)) {
                            foreach ($statement->from as $from) {
                                if ($from instanceof Expression) {
                                    $mapTable($from);
                                }
                            }
                        }

                        if (!empty($statement->join)) {
                            foreach ($statement->join as $join) {
                                if ($join instanceof JoinKeyword) {
                                    $join = $join->expr;
                                }

                                if ($join instanceof Expression) {
                                    $mapTable($join);
                                }
                            }
                        }

                        foreach ($statement->expr as $expr) {
                            if (($expr instanceof Expression)) {
                                $column = $expr->column;

                                if (empty($column) && str_contains($expr->expr ?: '', '*')) {
                                    $column = $expr->expr;
                                }

                                if (empty($column)) {
                                    continue;
                                }

                                $tableNonPrefix = str_replace(['#__', $tablePrefix], '', $expr->table ?: '');

                                if (empty($tableNonPrefix)) {
                                    $tableNonPrefix = array_keys($tablesMap)[0] ?? '';
                                }

                                if (empty($tableNonPrefix)
                                    || !($table = $tablesMap[$tableNonPrefix] ?? '')
                                    || !($class = TranslationHelper::getClassByTable($table))
                                ) {
                                    continue;
                                }

                                if (!isset($tablesMeta[$table])) {
                                    $tablesMeta[$table] = [
                                        'table'  => $table,
                                        'alias'  => $tableNonPrefix !== $table ? $tableNonPrefix : null,
                                        'fields' => [],
                                    ];
                                }

                                if (str_contains($column, '*')) {
                                    foreach ($class->getTranslationFields(true) as $tranField) {
                                        $tablesMeta[$table]['fields'][] = [null, $tranField];
                                    }

                                } else {
                                    $tablesMeta[$table]['fields'][] = [$expr->alias, $column];
                                }
                            }
                        }

                        // Parse where original value
                        if (count($tablesMeta)) {
                            /**
                             * @var TranslationExtensionAbstract $class
                             * @var DatabaseDriver               $db
                             */

                            $tablesMeta = array_values($tablesMeta);

                            if ($query instanceof DatabaseQuery) {
                                foreach ($tablesMeta as $i => $tableMeta) {
                                    $table  = $tableMeta['table'];
                                    $alias  = $tableMeta['alias'];
                                    $fields = $tableMeta['fields'];

                                    if (!($class = TranslationHelper::getClassByTable($table))) {
                                        continue;
                                    }

                                    $hasKey  = false;
                                    $keyName = $class->getKey();

                                    foreach ($fields as $field) {
                                        if ($field[1] === $keyName) {
                                            $hasKey = true;
                                            break;
                                        }
                                    }

                                    if (!$hasKey) {
                                        $key = ($alias ?? '#__' . $table) . '.' . $keyName;
                                        $query->select($this->quoteName($key));
                                        $tablesMeta[$i]['fields'][] = [null, $keyName];
                                    }
                                }
                            }

                            if (!empty($statement->where)) {
                                $originsMeta = [];

                                foreach ($statement->where as $where) {
                                    if (empty($where->identifiers) || empty($where->expr)) {
                                        continue;
                                    }

                                    $whereAlias  = $where->identifiers[0];
                                    $whereColumn = $where->identifiers[1] ?? $whereAlias;

                                    if (!($whereTable = $tablesMeta[$whereAlias] ?? null)) {
                                        $whereTable = array_keys($tablesMeta)[0];
                                    }

                                    // For current, just apply for the important things such as router...
                                    $allowFields = ['title', 'name', 'alias', 'path', 'route'];

                                    if ($whereTable && in_array($whereColumn, $allowFields)) {
                                        preg_match('/:([a-zA-Z0-9_-]+)/', $where->expr, $matches);
                                        $bindKey = $matches[1] ?? '';

                                        if ($bindKey) {
                                            $value = $boundedData[':' . $bindKey]->value ?? null;
                                        } else {
                                            preg_match('/\s*=\s*[\'"\[](.+)[\'"\]]/', $where->expr, $matches);
                                            $value = $matches[1] ?? null;

                                            if ($value) {
                                                $where->expr = str_replace($value, $this->quote($value), $where->expr);
                                            }
                                        }

                                        if ($value) {
                                            $originsMeta[$whereTable][] = [
                                                'field'   => $whereColumn,
                                                'value'   => $value,
                                                'bindKey' => $bindKey,
                                            ];
                                        }
                                    }
                                }

                                if (count($originsMeta)) {
                                    $db = Factory::getContainer()->get('org' . DatabaseInterface::class);

                                    foreach ($originsMeta as $table => $meta) {
                                        if (!($class = TranslationHelper::getClassByTable($table))) {
                                            continue;
                                        }

                                        $context = $class->getBaseContext() . '.%';
                                        $dbQuery = $db->getQuery(true)
                                            ->select($db->quoteName('context'))
                                            ->from($db->quoteName('#__joomlab_translations'))
                                            ->where($db->quoteName('language') . ' = :language')
                                            ->where($db->quoteName('context') . ' LIKE :context')
                                            ->bind(':language', $currentLang)
                                            ->bind(':context', $context);
                                        foreach ($meta as $metum) {
                                            $dbQuery->where('data->"$.' . $metum['field'] . '" = :' . $metum['field'])
                                                ->bind(':' . $metum['field'], $metum['value']);
                                        }

                                        if ($context = $db->setQuery($dbQuery)->loadResult()) {
                                            $parts = explode('.', $context);
                                            $id    = array_pop($parts);
                                            $dbQuery->clear()
                                                ->select($db->quoteName($metum['field']))
                                                ->from($db->quoteName('#__' . $table))
                                                ->where($db->quoteName($class->getKey()) . ' = :id')
                                                ->bind(':id', $id);

                                            if ($originValue = $db->setQuery($dbQuery)->loadResult()) {
                                                if ($metum['bindKey'] && $query instanceof DatabaseQuery) {
                                                    $query->bind(':' . $metum['bindKey'], $originValue);
                                                } elseif (is_string($query)) {
                                                    // TODO Update query string, maybe do nothing as we just support for DatabaseQuery only
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }

                } catch (Throwable $e) {
                }
            }

            // Reset tables meta
            $this->tablesMeta = $tablesMeta;

            return parent::setQuery($query, $offset, $limit);
        }

        protected function prepareStatement(string $query): StatementInterface
        {
            try {
                return new DatabaseStatement([
                    'statement'  => $this->connection->prepare($query, $this->options['driverOptions']),
                    'tablesMeta' => $this->tablesMeta,
                ]);
            } catch (Throwable $exception) {
                throw new PrepareStatementFailureException($exception->getMessage(), $exception->getCode(), $exception instanceof Exception ? $exception : null);
            }
        }
    }
}