<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_jltranslation
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\JLTranslation\Event;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormHelper;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Table\Table;
use Joomla\Event\Event;
use Joomla\Plugin\System\JLTranslation\Helper\TranslationHelper;
use Joomla\Plugin\System\Joomlab\Helper\LanguageHelper;
use Joomla\Plugin\System\Joomlab\Helper\StateHelper;
use Joomla\Registry\Registry;
use stdClass;

class RegisterEvent extends Event
{
    public const EVENT_NAME = 'onJLTranslationRegister';

    protected string $defaultLang = 'en-GB';

    protected bool $disableTranslateInline = false;

    protected array $disableTranslateInlineTables = [];

    protected Registry $params;

    protected function __construct()
    {
        $data = new RegisterEventData(
            prepareForm: ['onContentPrepareForm'],
            prepareData: ['onContentPrepareData'],
            prepareField: ['onCustomFieldsBeforePrepareField'],
        );

        parent::__construct(RegisterEvent::EVENT_NAME, ['subject' => $this, 'data' => $data]);
        $this->defaultLang                  = LanguageHelper::getDefaultLang();
        $this->params                       = ComponentHelper::getParams('com_jltranslation');
        $this->disableTranslateInline       = (bool)$this->params->get('disableTranslateInline', '0');
        $this->disableTranslateInlineTables = $this->params->get('disableTranslateInlineTables', []);
    }

    public function register(): void
    {
        static $registered = false;

        if ($registered) {
            return;
        }

        PluginHelper::importPlugin('jltranslation');
        $registered = true;
        $dispatcher = Factory::getApplication()->getDispatcher();
        $data       = $dispatcher
            ->dispatch(RegisterEvent::EVENT_NAME, $this)
            ->getArgument('data', []);

        foreach ($data as $method => $events) {
            foreach ($events as $event) {
                if (is_array($event)
                    && is_string($event[0])
                    && is_callable($event[1])
                ) {
                    $dispatcher->addListener($event[0], $event[1]);
                } elseif (is_string($event)) {
                    $dispatcher->addListener($event, [$this, $method]);
                }
            }
        }
    }

    public function prepareForm(Event $event)
    {
        $form = $event->getArgument('subject', $event->getArgument(0));
        $data = $event->getArgument('data', $event->getArgument(1));
        $app  = Factory::getApplication();

        if (!$app->isClient('administrator')
            || $app->getInput()->get('option') === 'com_jltranslation'
            || !($form instanceof Form)
            || !($class = TranslationHelper::getClassByContext($form->getName()))
            || !Multilanguage::isEnabled()
            || ($this->disableTranslateInline && (empty($this->disableTranslateInlineTables) || in_array($class->getReferenceTable(), $this->disableTranslateInlineTables)))
        ) {
            return;
        }

        $class->setForm($form);
        FormHelper::addFieldPrefix('Joomla\\Plugin\\System\\Joomlab\\Form\\Field');
        $key = $class->getKey();

        if (!$form->getValue($key)) {
            $pk = ((array)$data)[$key] ?? null;

            if ($pk) {
                $form->setValue($key, null, $pk);
            }
        }

        $id             = $form->getValue($key);
        $transformField = function (string $fieldName, ?string $group = null) use ($form) {
            $fieldType = $form->getFieldAttribute($fieldName, 'type', 'text', $group);
            $form->setFieldAttribute($fieldName, 'type', 'translation', $group);
            $form->setFieldAttribute($fieldName, 'ignoreDefault', 'true', $group);
            $form->setFieldAttribute($fieldName, 'fieldType', $fieldType, $group);
        };

        foreach ($class->getTranslationFields() as $field) {
            $parts = explode('.', $field, 2);
            $group = count($parts) > 1 ? $parts[0] : null;
            $name  = $group ? $parts[1] : $parts[0];
            $transformField($name, $group);
        }

        if ($fields = $form->getGroup('com_fields')) {
            $config            = ComponentHelper::getParams('com_jltranslation');
            $excludeFieldNames = preg_split('/\s*,\s*/', $config->get('excludeFieldNames', ''), -1, PREG_SPLIT_NO_EMPTY);
            $validFieldTypes   = preg_split('/\s*,\s*/', $config->get('validFieldTypes', 'calendar,color,editor,integer,imagelist,accessiblemedia,text,textarea,url'), -1, PREG_SPLIT_NO_EMPTY);

            foreach ($fields as $field) {
                if (in_array(strtolower($field->type), $validFieldTypes) && !in_array($field->fieldname, $excludeFieldNames)) {
                    $transformField($field->__get('fieldname'), 'com_fields');
                }
            }
        }

        if ($id) {
            $fieldsData             = new stdClass();
            $fieldsData->id         = $id;
            $fieldsData->com_fields = $form->getValue('com_fields', null, []);
            $this->_prepareData($form->getName(), $fieldsData);
        }
    }

    protected function _prepareData($context, $data): void
    {
        $app = Factory::getApplication();

        if (!$app->isClient('administrator')
            || $app->getInput()->get('option') === 'com_jltranslation'
            || !Multilanguage::isEnabled()
            || (!is_array($data) && !is_object($data))
            || !($class = TranslationHelper::getClassByContext($context))
            || empty($pk = ((array)$data) [$class->getKey()] ?? 0)
            || ($this->disableTranslateInline && (empty($this->disableTranslateInlineTables) || in_array($class->getReferenceTable(), $this->disableTranslateInlineTables)))
        ) {
            return;
        }

        $originData   = new Registry($data);
        $transData    = new Registry(TranslationHelper::getTranslationData($class->getBaseContext() . '.' . $pk));
        $parseDefault = function ($defaultValue) {
            if ($defaultValue instanceof Registry) {
                $langValue = new Registry();
            } elseif (is_object($defaultValue)) {
                $langValue = new stdClass();
            } elseif (is_array($defaultValue)) {
                $langValue = [];
            } else {
                $langValue = '';
            }

            return $langValue;
        };

        $state        = StateHelper::getInstance();
        $stateContext = 'translation.' . TranslationHelper::prepareContext($context, true) . '.' . $pk;
        $fields       = array_merge($class->getTranslationFields(true), ['com_fields']);
        $langCodes    = LanguageHelper::getLanguageCodes();

        foreach ($fields as $field) {
            if (!$originData->exists($field)) {
                continue;
            }

            // Revert default value;
            $defaultValue = $originData->get($field);
            $state->set($stateContext . '.' . $this->defaultLang . '.' . $field, $defaultValue);

            foreach ($langCodes as $langCode) {
                if ($langCode === $this->defaultLang) {
                    continue;
                }

                $path = $langCode . '.' . $field;

                if ($transData->exists($path)) {
                    $state->set($stateContext . '.' . $path, $transData->get($path));
                } else {
                    $state->set($stateContext . '.' . $path, $parseDefault($defaultValue));
                }
            }
        }
    }

    public static function getInstance(): RegisterEvent
    {
        static $instance = null;

        if (null === $instance) {
            $instance = new static();
        }

        return $instance;
    }

    public function prepareData(Event $event)
    {
        $context = $event->getArgument('context', $event->getArgument(0));
        $data    = $event->getArgument('data', $event->getArgument(1));
        $this->_prepareData($context, $data);
    }

    public function afterSave(Event $event): void
    {
        $context = $event->getArgument('context', $event->getArgument(0));
        $table   = $event->getArgument('subject', $event->getArgument(1));

        if (!$table instanceof Table) {
            foreach ($event->getArguments() as $argument) {
                if ($argument instanceof Table) {
                    $table = $argument;
                    break;
                }
            }
        }

        if (!is_string($context)
            && empty($context)
            && $table instanceof Table
            && ($class = TranslationHelper::getClassByTable(str_replace('#__', '', $table->getTableName())))
        ) {
            $context = 'com_' . strtolower(str_replace('\\', '.', $class->getBaseContext()));
        }

        $app              = Factory::getApplication();
        $postTranslations = $app->getInput()->post->get('JLTranslationFilteredData', [], 'ARRAY') ?? [];

        if (!is_string($context)
            || !($class = TranslationHelper::getClassByContext($context))
            || !Multilanguage::isEnabled()
            || !$table instanceof Table
            || empty($postTranslations)
            || ($this->disableTranslateInline && (empty($this->disableTranslateInlineTables) || in_array($class->getReferenceTable(), $this->disableTranslateInlineTables)))
        ) {
            return;
        }

        $itemData     = new Registry($table);
        $pk           = $itemData->get($class->getKey(), 0);
        $transContext = $class->getBaseContext() . '.' . $pk;

        foreach ($postTranslations as $langCode => $translation) {
            // Check to ignore default value
            if ($langCode === $this->defaultLang) {
                continue;
            }

            $class->prepareSaveData($transContext, $langCode, $table, $translation);
            $table = $app
                ->bootComponent('com_jltranslation')
                ->getMVCFactory()
                ->createTable('Translation', 'Administrator');

            // Try to load the current data first
            $table->load(['context' => $transContext, 'language' => $langCode]);
            $table->bind(['context' => $transContext, 'language' => $langCode, 'data' => $translation]);
            $table->store();
        }
    }

    public function prepareField(Event $event): void
    {
        $context = $event->getArgument('context', $event->getArgument(0));
        $item    = $event->getArgument('item', $event->getArgument(1));
        $field   = $event->getArgument('subject', $event->getArgument(2));

        if (!$this->isTranslatable()
            || !($class = TranslationHelper::getClassByContext($context))
            || !is_object($item)
            || !is_object($field)
            || empty($pk = $item->{$class->getKey()})
            || empty($field->name)
        ) {
            return;
        }

        $activeLang      = Factory::getApplication()->getLanguage()->getTag();
        $translationData = TranslationHelper::getTranslationData($class->getBaseContext() . '.' . $pk, $activeLang);

        if (isset($translationData['com_fields'][$field->name])) {
            $o                 = new stdClass();
            $o->{$field->name} = $field->value;
            $translation       = [$field->name => $translationData['com_fields'][$field->name]];
            $class->prepareDisplayData($translation, $o);
            $field->value = $o->{$field->name};
        }
    }

    public function isTranslatable(): bool
    {
        $app = Factory::getApplication();

        return Multilanguage::isEnabled()
            && $app->isClient('site')
            && $app->getLanguage()->getTag() !== $this->defaultLang;
    }

    public function getEventData(): RegisterEventData
    {
        return $this->getArgument('data');
    }
}