<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_jltranslation
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\JLTranslation\Extension;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Router;
use Joomla\CMS\Router\SiteRouter;
use Joomla\CMS\Toolbar\Toolbar;
use Joomla\CMS\User\UserFactoryAwareTrait;
use Joomla\CMS\Version;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseInterface;
use Joomla\Event\SubscriberInterface;
use Joomla\Plugin\System\JLTranslation\Driver\DatabaseDriver;
use Joomla\Plugin\System\JLTranslation\Event\RegisterEvent;
use Joomla\Plugin\System\JLTranslation\Event\TranslationToolbarButtonEvent;
use Joomla\Plugin\System\JLTranslation\Helper\TranslationHelper;
use Joomla\Plugin\System\Joomlab\Helper\AssetHelper;
use Joomla\Plugin\System\Joomlab\Helper\BackwardCompatibilityHelper;
use Joomla\Plugin\System\Joomlab\Helper\LanguageHelper;

// Force language filter load before
PluginHelper::importPlugin('system', 'languagefilter');

// Force Joomlab CORE and Joomla fields plugin loads after that
PluginHelper::importPlugin('system', 'joomlab');

if (PluginHelper::isEnabled('system', 'fields') && Factory::getApplication()->isClient('administrator')) {
    PluginHelper::importPlugin('system', 'fields');
}

final class JLTranslation extends CMSPlugin implements SubscriberInterface
{
    use DatabaseAwareTrait;
    use UserFactoryAwareTrait;

    public static function getSubscribedEvents(): array
    {
        return [
            'onAfterInitialise'         => 'onAfterInitialise',
            'onAfterRoute'              => 'onAfterRoute',
            'onAfterInitialiseDocument' => 'onAfterInitialiseDocument',
        ];
    }

    public function onAfterInitialise()
    {
        $app = Factory::getApplication();

        if (!class_exists(DatabaseDriver::class) || !Multilanguage::isEnabled()) {
            return;
        }

        // Register event
        RegisterEvent::getInstance()->register();

        if ($app->isClient('site')) {
            $config  = $app->getConfig();
            $options = [
                'host'     => $config->get('host'),
                'user'     => $config->get('user'),
                'password' => $config->get('password'),
                'database' => $config->get('db'),
                'prefix'   => $config->get('dbprefix'),
                'dbType'   => $config->get('dbtype'),
                'debug'    => $config->get('debug'),
            ];

            $orgDatabase       = $this->getDatabase();
            $database          = new DatabaseDriver($options);
            Factory::$database = $database;
            Factory::getContainer()
                ->alias(DatabaseInterface::class, DatabaseDriver::class)
                ->alias('DatabaseDriver', DatabaseDriver::class)
                ->alias('db', DatabaseDriver::class)
                ->share(DatabaseInterface::class, $database)
                ->share('org' . DatabaseInterface::class, $orgDatabase);

            /** @var SiteRouter $router */
            $router       = Factory::getContainer()->get(SiteRouter::class);
            $siteMenu     = $app->getMenu();
            $menuReloaded = false;
            $reloadMenu   = function () use (&$menuReloaded, $siteMenu, $database) {
                if (!$menuReloaded && Factory::getApplication()->getLanguage()->getTag() !== LanguageHelper::getDefaultLang()) {
                    $menuReloaded = true;
                    $siteMenu->__construct(['db' => $database]);
                    $siteMenu->load();
                }
            };

            $router->attachParseRule($reloadMenu, Router::PROCESS_BEFORE);
            $router->attachBuildRule($reloadMenu, Router::PROCESS_BEFORE);
        }
    }

    public function onAfterRoute()
    {
        if (Version::MAJOR_VERSION === 4) {
            $app = $this->getApplication();

            if (!$app->getDocument()) {
                $app->loadDocument();
            }

            $this->onAfterInitialiseDocument();
        }
    }

    public function onAfterInitialiseDocument()
    {
        $app   = $this->getApplication();
        $input = $app->getInput();

        if (!$app->isClient('administrator')
            || !Multilanguage::isEnabled()
            || $input->getMethod() !== 'GET'
        ) {
            return;
        }

        $component = $input->getCmd('option');
        $extClass  = null;
        $pk        = 0;

        foreach (TranslationHelper::bootExtensions() as $className => $classObj) {
            /** @var TranslationExtensionAbstract $classObj */
            $parts = explode('\\', $className);
            [$view, $id] = $classObj->getEditViewId($input);

            if ($component === $classObj->getComponentName() && strtolower($parts[count($parts) - 1]) === $view) {
                if ($id < 1) {
                    return;
                }

                $extClass = $classObj;
                $pk       = (int)$id;
                break;
            }
        }

        if (!$extClass) {
            return;
        }

        HTMLHelper::_('joomlab.framework');

        if (PluginHelper::isEnabled('jltranslation', 'pro')) {
            PluginHelper::importPlugin('jltranslation', 'pro');
            $app->getDispatcher()
                ->dispatch(
                    TranslationToolbarButtonEvent::EVENT_NAME,
                    new TranslationToolbarButtonEvent(['subject' => $extClass, 'pk' => $pk])
                );
        } else {
            AssetHelper::getWebAssetManager()
                ->registerAndUseScript('com_jltranslation.translation', 'com_jltranslation/translation.min.js');
        }

        $defaultLang = LanguageHelper::getDefaultLang();
        $app->getLanguage()->load('com_jltranslation', JPATH_ADMINISTRATOR . '/components/com_jltranslation');
        $toolbar    = BackwardCompatibilityHelper::getToolbar();
        $transGroup = $toolbar->dropdownButton('translation-group');
        $context    = $extClass->getBaseContext() . '.' . $pk;
        $transGroup->configure(
            function (Toolbar $childBar) use ($context, $defaultLang) {
                $childBar->standardButton('jl-translation-languages', Text::_('COM_JLTRANSLATION_TRANSLATIONS'))
                    ->setOption('icon', 'fa fa-language')
                    ->setOption('onclick', 'void(0)');

                foreach (LanguageHelper::getLanguageCodes() as $langCode) {
                    if ($langCode === $defaultLang) {
                        continue;
                    }

                    $text = LanguageHelper::langCodeToFlag($langCode) . ' ' . $langCode;

                    if (TranslationHelper::isEnabled($context, $langCode)) {
                        $text .= ' <i class="text-primary fa fa-check-circle"></i>';
                    } else {
                        $text .= ' <i class="text-danger fa fa-times-circle"></i>';
                    }

                    $childBar->standardButton('jl-translation-language-' . $langCode, $text)
                        ->setOption('onclick', 'void(0)');
                }
            }
        );
    }
}
