<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_jltranslation
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\JLTranslation\Helper;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Database\DatabaseInterface;
use Joomla\Plugin\System\JLTranslation\Event\ExtensionEvent;
use Joomla\Plugin\System\JLTranslation\Extension\Banners\Banner;
use Joomla\Plugin\System\JLTranslation\Extension\Categories\Category;
use Joomla\Plugin\System\JLTranslation\Extension\Contact\Contact;
use Joomla\Plugin\System\JLTranslation\Extension\Content\Article;
use Joomla\Plugin\System\JLTranslation\Extension\Fields\Field;
use Joomla\Plugin\System\JLTranslation\Extension\Fields\Group as FieldGroup;
use Joomla\Plugin\System\JLTranslation\Extension\Menus\Item;
use Joomla\Plugin\System\JLTranslation\Extension\Modules\Module;
use Joomla\Plugin\System\JLTranslation\Extension\Tags\Tag;
use Joomla\Plugin\System\JLTranslation\Extension\TranslationExtensionAbstract;
use Joomla\Plugin\System\JLTranslation\Extension\TranslationExtensionPrepareStatementInterface;
use Joomla\Plugin\System\JLTranslation\Extension\Users\Group as UserGroup;
use Joomla\Plugin\System\Joomlab\Helper\LanguageHelper;
use stdClass;

class TranslationHelper
{
    protected static array $enabledTranslations = [];

    public static function getDashboardCardData(?string $table = null, ?int $id = null): array
    {
        /**
         * @var DatabaseInterface            $db
         * @var TranslationExtensionAbstract $class
         */
        $items           = [];
        $defaultSiteLang = ComponentHelper::getParams('com_languages')->get('site', 'en-GB');
        $languages       = array_filter(LanguageHelper::getLanguageCodes(), function ($language) use ($defaultSiteLang) {
            return $defaultSiteLang !== $language;
        });

        foreach (static::bootExtensions() as $class) {
            $refTable = $class->getReferenceTable();

            if ($table && $table !== $refTable) {
                continue;
            }

            $db    = $class->getDatabase();
            $query = $db->getQuery(true)
                ->select('COUNT(*)')
                ->from($db->quoteName('#__' . $refTable));
            $item  = [
                'title'     => $class->getTitle(),
                'total'     => $db->setQuery($query)->loadResult(),
                'table'     => $refTable,
                'component' => $class->getComponentName(),
                'count'     => [],

            ];

            foreach ($languages as $language) {
                $context = $class->getBaseContext() . ($id ? '.' . $id : '.%');
                $query->clear()
                    ->select('COUNT(*)')
                    ->from($db->quoteName('#__joomlab_translations'))
                    ->where($db->quoteName('context') . ' LIKE :context')
                    ->where($db->quoteName('state') . ' = 1')
                    ->where($db->quoteName('language') . ' = :language')
                    ->bind(':context', $context)
                    ->bind(':language', $language);
                $item['count'][$language]['enabled'] = $db->setQuery($query)->loadResult();

                $query->clear()
                    ->select('COUNT(*)')
                    ->from($db->quoteName('#__joomlab_translations'))
                    ->where($db->quoteName('context') . ' LIKE :context')
                    ->where($db->quoteName('state') . ' = 0')
                    ->where($db->quoteName('language') . ' = :language')
                    ->bind(':context', $context)
                    ->bind(':language', $language);
                $item['count'][$language]['disabled'] = $db->setQuery($query)->loadResult();
            }

            if ($table) {
                return $item;
            }

            $items[] = $item;
        }

        usort($items, function ($a, $b) {
            return $a['title'] < $b['title'] ? -1 : 1;
        });

        return $items;
    }

    public static function bootExtensions(): array
    {
        static $extensions = null;

        if (null === $extensions) {
            PluginHelper::importPlugin('jltranslation');
            $coreExtensions = [
                Category::class,
                Banner::class,
                Article::class,
                Contact::class,
                Field::class,
                FieldGroup::class,
                Item::class,
                Module::class,
                Tag::class,
                UserGroup::class,
            ];
            $classes        = Factory::getApplication()
                ->getDispatcher()
                ->dispatch(ExtensionEvent::EVENT_NAME, new ExtensionEvent($coreExtensions))
                ->getArgument('subject')
                ->data;
            /** @var DatabaseInterface $db */
            $extensions = [];
            $db         = Factory::getContainer()->get(DatabaseInterface::class);
            $listTables = array_map(function ($table) use ($db) {
                return str_replace($db->getPrefix(), '', $table);
            }, $db->getTableList());

            foreach ($classes as $className) {
                if (!class_exists($className) || isset($extensions[$className])) {
                    continue;
                }

                $class = new $className();

                if (($class instanceof TranslationExtensionAbstract) && in_array($class->getReferenceTable(), $listTables)) {
                    $extensions[$className] = $class;
                }
            }
        }

        return $extensions;
    }

    public static function prepareStatementFetchResult(array|object &$result): void
    {
        // For UCM content/tags/version
        if (is_object($result)
            && !empty($result->core_content_id)
            && !empty($result->content_item_id)
            && !empty($result->type_alias)
            && !empty($result->core_title)
            && !empty($result->core_alias)
            && ($class = static::getClassByContext($result->type_alias))
        ) {
            $item        = new stdClass();
            $item->id    = $result->content_item_id;
            $item->title = $result->core_title;
            $item->alias = $result->core_alias;

            static::translate($class->getReferenceTable(), $item);
            $result->core_title = $item->title;
            $result->core_alias = $item->alias;
        }

        foreach (static::bootExtensions() as $class) {
            /** @var TranslationExtensionAbstract $class */

            if ($class instanceof TranslationExtensionPrepareStatementInterface) {
                $class->prepareStatementFetchResult($result);
            }
        }
    }

    public static function getClassByContext(string $context): ?TranslationExtensionAbstract
    {
        $paths = implode('\\',
            array_map(function ($part) {
                return ucfirst(str_replace('com_', '', strtolower($part)));
            }, explode('.', TranslationHelper::prepareContext($context)))
        );

        $coreClass    = 'Joomla\\Plugin\\System\\JLTranslation\\Extension\\' . $paths;
        $pluginPrefix = 'Joomla\\Plugin\\JLTranslation\\';

        foreach (static::bootExtensions() as $className => $class) {
            if ($className === $coreClass || (str_starts_with($className, $pluginPrefix) && str_ends_with($className, $paths))) {
                return $class;
            }
        }

        return null;
    }

    public static function prepareContext(string $context, bool $checkContext = false): string
    {
        $input = Factory::getApplication()->getInput();

        if (str_starts_with($context, 'com_categories.category')) {
            return 'com_categories.category' . ($checkContext ? $input->get('extension', '') : '');
        }

        if (str_starts_with($context, 'com_fields.field')) {
            return 'com_fields.field' . ($checkContext ? '.' . $input->get('context', '') : '');
        }

        if (str_starts_with($context, 'com_fields.group')) {
            return 'com_fields.group' . ($checkContext ? '.' . $input->get('context', '') : '');
        }

        return $context;
    }

    public static function translate(string $table, object $item): void
    {
        if (!($class = static::getClassByTable($table))) {
            return;
        }

        $pk = $item->{$class->getKey()} ?? null;

        if (!$pk) {
            return;
        }

        static $translated = [];
        $context = $class->getBaseContext() . '.' . $pk;

        if (!array_key_exists($context, $translated)) {
            $lang  = Factory::getApplication()->getLanguage()->getTag();
            $db    = Factory::getContainer()->get('org' . DatabaseInterface::class);
            $query = $db->getQuery(true)
                ->select($db->quoteName('data'))
                ->from($db->quoteName('#__joomlab_translations'))
                ->where($db->quoteName('context') . ' = :context')
                ->where($db->quoteName('language') . ' = :language')
                ->where($db->quoteName('state') . ' = 1')
                ->bind(':context', $context)
                ->bind(':language', $lang);
            $translated[$context] = json_decode($db->setQuery($query)->loadResult() ?: '{}', true);
        }

        $class->prepareDisplayData($translated[$context], $item);
    }

    public static function getClassByTable(string $table): ?TranslationExtensionAbstract
    {
        foreach (static::bootExtensions() as $class) {
            if ($table === $class->getReferenceTable()) {
                return $class;
            }
        }

        return null;
    }

    public static function isEnabled(string $context, string $language, $ignoreNoneExists = false): bool
    {
        static::getTranslationData($context, $language);

        if ($ignoreNoneExists && !isset(static::$enabledTranslations[$context][$language])) {
            return true;
        }

        return !empty(static::$enabledTranslations[$context][$language]);
    }

    public static function getTranslationData(string $context, ?string $activeLang = null): array
    {
        $dataKey = $context . '.' . ($activeLang ?? '*');
        static $data = [];

        if (!array_key_exists($dataKey, $data)) {
            $db    = Factory::getContainer()->get(DatabaseInterface::class);
            $query = $db->getQuery(true)
                ->select($db->quoteName(['language', 'data', 'state']))
                ->from($db->quoteName('#__joomlab_translations'))
                ->where($db->quoteName('context') . ' = :context')
                ->bind(':context', $context);

            if ($activeLang) {
                $query->where($db->quoteName('language') . ' = :language')
                    ->bind(':language', $activeLang);
            }

            $data[$dataKey] = [];

            foreach ($db->setQuery($query)->loadObjectList() as $row) {
                $transData                                        = json_decode($row->data ?: '{}', true);
                $language                                         = $row->language;
                static::$enabledTranslations[$context][$language] = !empty($row->state);

                if ($activeLang && $language === $activeLang) {
                    $data[$dataKey] = $transData;
                    break;
                }

                foreach ($transData as $k => $v) {
                    $data[$dataKey][$language][$k] = $v;
                }
            }
        }

        return $data[$dataKey];
    }
}
