<?php

/**
 * @package     Joomlab
 * @subpackage  com_jladmin
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\Joomlab\Helper;

defined('_JEXEC') or die;

use FilesystemIterator;
use Joomla\Archive\Zip;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Path;
use RecursiveCallbackFilterIterator;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use RuntimeException;
use SplFileInfo;
use ZipArchive;

class ZipHelper
{
    private const CTRL_DIR_HEADER = "\x50\x4b\x01\x02";

    private const CTRL_DIR_END = "\x50\x4b\x05\x06\x00\x00\x00\x00";

    private const FILE_HEADER = "\x50\x4b\x03\x04";

    private array $contents = [];

    private array $ctrlDir = [];

    private readonly bool $nativeSupport;

    private ?ZipArchive $zipArchive = null;

    public function __construct(private readonly string $filePath)
    {
        $this->nativeSupport = Zip::hasNativeSupport();

        if ($this->nativeSupport) {
            $this->zipArchive = new ZipArchive();

            if ($this->zipArchive->open($filePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
                throw new RuntimeException('Could not open ZIP file for writing.');
            }
        }
    }

    public static function extract(string $archive, ?string $destination = null): bool
    {
        if (!is_file($archive)) {
            throw new RuntimeException('The compressed file does not exist: ' . $archive);
        }

        if ('zip' !== strtolower(FileSystemHelper::getExt($archive))) {
            throw new RuntimeException('The compressed file is not a zip file: ' . $archive);
        }

        if (!$destination) {
            $destination = $archive . '-zip';
        }

        return (new Zip())->extract($archive, $destination);
    }

    public static function compress(string $archive, string $path): void
    {
        if (!is_dir($path)) {
            throw new RuntimeException('Directory does not exist: ' . $path);
        }

        if (!str_ends_with($archive, '.zip')) {
            $archive .= '.zip';
        }

        /** @var SplFileInfo $file */
        $zip      = new ZipHelper($archive);
        $iterator = new RecursiveDirectoryIterator($path, FilesystemIterator::SKIP_DOTS);
        $filter   = new RecursiveCallbackFilterIterator($iterator, fn(SplFileInfo $file) => $file->isFile() && $file->isReadable());
        $files    = new RecursiveIteratorIterator($filter);

        foreach ($files as $file) {
            $zip->addFile([
                'name' => preg_replace('#' . preg_quote(JPATH_ROOT, '#') . '/#', '', Path::clean($file->getRealPath(), '/')),
                'data' => FileSystemHelper::readFile($file),
                'time' => $file->getATime(),
            ]);
        }

        if (!$zip->close()) {
            throw new RuntimeException('Failed to create zip file: ' . $archive);
        }
    }

    public function addFile(array $file): static
    {
        $hasPathFile = isset($file['file']) && is_file($file['file']);

        if ($this->nativeSupport) {
            if (($hasPathFile && !$this->zipArchive->addFile($file['file'], $file['name']))
                || (!$hasPathFile && !$this->zipArchive->addFromString($file['name'], $file['data']))
            ) {
                throw new RuntimeException('Could not add backup file: ' . $file['name']);
            }

            return $this;
        }

        if ($hasPathFile) {
            $file['data'] = file_get_contents($file['file']);
        }

        $data = &$file['data'];
        $name = str_replace('\\', '/', $file['name']);

        // See if time/date information has been provided.
        $fTime = null;

        if (isset($file['time'])) {
            $fTime = $file['time'];
        }

        // Get the hex time.
        $dTime    = dechex($this->unix2DosTime($fTime));
        $hexdTime = chr(hexdec($dTime[6] . $dTime[7])) . chr(hexdec($dTime[4] . $dTime[5])) . chr(hexdec($dTime[2] . $dTime[3]))
            . chr(hexdec($dTime[0] . $dTime[1]));

        // Begin creating the ZIP data.
        $fr = self::FILE_HEADER;

        // Version needed to extract.
        $fr .= "\x14\x00";

        // General purpose bit flag.
        $fr .= "\x00\x00";

        // Compression method.
        $fr .= "\x08\x00";

        // Last modification time/date.
        $fr .= $hexdTime;

        // "Local file header" segment.
        $uncLen = strlen($data);
        $crc    = crc32($data);
        $zData  = gzcompress($data);
        $zData  = substr(substr($zData, 0, -4), 2);
        $cLen   = strlen($zData);

        // CRC 32 information.
        $fr .= pack('V', $crc);

        // Compressed filesize.
        $fr .= pack('V', $cLen);

        // Uncompressed filesize.
        $fr .= pack('V', $uncLen);

        // Length of filename.
        $fr .= pack('v', strlen($name));

        // Extra field length.
        $fr .= pack('v', 0);

        // File name.
        $fr .= $name;

        // "File data" segment.
        $fr .= $zData;

        // Add this entry to array.
        $oldOffset        = strlen(implode('', $this->contents));
        $this->contents[] = &$fr;

        // Add to central directory record.
        $cdRec = self::CTRL_DIR_HEADER;

        // Version made by.
        $cdRec .= "\x00\x00";

        // Version needed to extract
        $cdRec .= "\x14\x00";

        // General purpose bit flag
        $cdRec .= "\x00\x00";

        // Compression method
        $cdRec .= "\x08\x00";

        // Last mod time/date.
        $cdRec .= $hexdTime;

        // CRC 32 information.
        $cdRec .= pack('V', $crc);

        // Compressed filesize.
        $cdRec .= pack('V', $cLen);

        // Uncompressed filesize.
        $cdRec .= pack('V', $uncLen);

        // Length of filename.
        $cdRec .= pack('v', strlen($name));

        // Extra field length.
        $cdRec .= pack('v', 0);

        // File comment length.
        $cdRec .= pack('v', 0);

        // Disk number start.
        $cdRec .= pack('v', 0);

        // Internal file attributes.
        $cdRec .= pack('v', 0);

        // External file attributes -'archive' bit set.
        $cdRec .= pack('V', 32);

        // Relative offset of local header.
        $cdRec .= pack('V', $oldOffset);

        // File name.
        $cdRec .= $name;

        // Save to central directory array.
        $this->ctrlDir[] = &$cdRec;

        return $this;
    }

    private function unix2DosTime($unixTime = null)
    {
        $timeArray = $unixTime === null ? getdate() : getdate($unixTime);

        if ($timeArray['year'] < 1980) {
            $timeArray['year']    = 1980;
            $timeArray['mon']     = 1;
            $timeArray['mday']    = 1;
            $timeArray['hours']   = 0;
            $timeArray['minutes'] = 0;
            $timeArray['seconds'] = 0;
        }

        return (($timeArray['year'] - 1980) << 25) | ($timeArray['mon'] << 21) | ($timeArray['mday'] << 16) | ($timeArray['hours'] << 11) |
            ($timeArray['minutes'] << 5) | ($timeArray['seconds'] >> 1);
    }

    public function close(): bool
    {
        if ($this->nativeSupport) {
            return $this->zipArchive->close();
        }

        $data = implode('', $this->contents);
        $dir  = implode('', $this->ctrlDir);

        /*
         * Buffer data:
         * Total # of entries "on this disk".
         * Total # of entries overall.
         * Size of central directory.
         * Offset to start of central dir.
         * ZIP file comment length.
         */
        $buffer = $data . $dir . self::CTRL_DIR_END .
            pack('v', count($this->ctrlDir)) .
            pack('v', count($this->ctrlDir)) .
            pack('V', strlen($dir)) .
            pack('V', strlen($data)) .
            "\x00\x00";

        return File::write($this->filePath, $buffer);
    }
}
