<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_jltranslation
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\JLTranslation\Extension\Content;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Router\Route;
use Joomla\Component\Content\Site\Helper\RouteHelper;
use Joomla\Database\QueryInterface;
use Joomla\Plugin\System\JLTranslation\Extension\TranslationExtensionAbstract;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

use function count;
use function is_array;

class Article extends TranslationExtensionAbstract
{
    public function prepareDisplayData(array $translation, object $item): void
    {
        parent::prepareDisplayData($translation, $item);
        $text = trim($translation['articletext'] ?? '');

        if (!empty($text)) {
            $pattern = '#<hr\s+id=(["\'])system-readmore(["\'])\s*/*>#i';
            $tagPos  = preg_match($pattern, $text);

            if ($tagPos == 0) {
                $introtext = $text;
                $fulltext  = '';
            } else {
                [$introtext, $fulltext] = preg_split($pattern, $text, 2);
            }

            $item->introtext = $introtext;
            $item->fulltext  = $fulltext;
        }

        // Override article text
        if (isset($item->params)) {
            if ((new Registry($item->params))->get('show_intro', '1') == '1') {
                $item->text = $item->introtext . ' ' . $item->fulltext;
            } elseif ($item->fulltext) {
                $item->text = $item->fulltext;
            } else {
                $item->text = $item->introtext;
            }
        }

        // Override article slug
        if (isset($item->alias)) {
            $item->slug = $item->alias ? ($item->id . ':' . $item->alias) : $item->id;
        }

        // Override article link
        if (isset($item->slug) && isset($item->catid) && isset($item->language)) {
            $item->readmore_link = Route::_(RouteHelper::getArticleRoute($item->slug, $item->catid, $item->language));

            if (!isset($item->link)) {
                $item->link = $item->readmore_link;
            }
        }
    }

    public function registerTranslationFields(): array
    {
        return [
            'title',
            'alias',
            'articletext',
            'metadesc',
            'metakey',
            'metadata.author',
            'metadata.rights',
            'images.image_intro',
            'images.image_intro_alt',
            'images.image_fulltext',
            'images.image_fulltext_alt',
            'images.image_fulltext_caption',
            'urls.urla',
            'urls.urlatext',
            'urls.urlb',
            'urls.urlbtext',
            'urls.urlc',
            'urls.urlctext',
            'attribs.alternative_readmore',
            'attribs.article_page_title',
        ];
    }

    public function getTitleField(): ?string
    {
        return 'title';
    }

    public function getReferenceTable(): string
    {
        return 'content';
    }

    public function prepareListQuery(QueryInterface $query): void
    {
        $categoryId = $this->translationsModel->getState('filter.category_id', []);

        if (!is_array($categoryId)) {
            $categoryId = $categoryId ? [$categoryId] : [];
        }

        if (count($categoryId)) {
            $categoryId       = ArrayHelper::toInteger($categoryId);
            $categoryTable    = Factory::getApplication()
                ->bootComponent('com_categories')
                ->getMVCFactory()
                ->createTable('Category', 'Administrator');
            $subCatItemsWhere = [];
            $db               = $this->getDatabase();
            $query->join('INNER', $db->quoteName('#__categories', 'c'), 'a.catid = c.id');

            foreach ($categoryId as $cid) {
                $categoryTable->load($cid);
                $valuesToBind       = [$categoryTable->lft, $categoryTable->rgt];
                $bounded            = $query->bindArray($valuesToBind);
                $categoryWhere      = $db->quoteName('c.lft') . ' >= ' . $bounded[0] . ' AND ' . $db->quoteName('c.rgt') . ' <= ' . $bounded[1];
                $subCatItemsWhere[] = '(' . $categoryWhere . ')';
            }

            $query->where('(' . implode(' OR ', $subCatItemsWhere) . ')');
        }
    }
}