<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_jltranslation
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\JLTranslation\Extension;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Filter\InputFilter;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Language\Text;
use Joomla\Component\JLTranslation\Administrator\Model\TranslationsModel;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\QueryInterface;
use Joomla\Input\Input;
use Joomla\Plugin\System\JLTranslation\Event\RegisterTranslationFieldsEvent;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;
use ReflectionClass;
use stdClass;

abstract class TranslationExtensionAbstract
{
    protected ?Form $form = null;

    protected ?TranslationsModel $translationsModel = null;

    public function setTranslationsModel(TranslationsModel $model): static
    {
        $this->translationsModel = $model;

        return $this;
    }

    abstract public function getReferenceTable(): string;

    public function prepareTranslationForm(): Form
    {
        $parts     = $this->extract();
        $component = 'com_' . $parts[0];
        $model     = $parts[1];
        $app       = Factory::getApplication();
        $language  = $app->getLanguage();
        Form::addFieldPath(JPATH_ADMINISTRATOR . '/components/' . $component . '/models/fields');
        Form::addFormPath(JPATH_ADMINISTRATOR . '/components/' . $component . '/models/forms');
        Form::addFormPath(JPATH_ADMINISTRATOR . '/components/' . $component . '/forms');
        $language->load($component, JPATH_ADMINISTRATOR)
        || $language->load($component, JPATH_ADMINISTRATOR . '/components/' . $component);

        return $app
            ->bootComponent($component)
            ->getMVCFactory()
            ->createModel(ucfirst($model), 'Administrator')
            ->getForm();
    }

    protected function extract(): array
    {
        static $extracts = [];
        $className = get_class($this);

        if (!isset($extracts[$className])) {
            $extracts[$className] = explode('\\', strtolower(preg_replace('/^.*\\\\Extension\\\\/', '', $className)));
        }

        return $extracts[$className];
    }

    public function prepareFilterForm(Form $form): void
    {
        $extract   = $this->extract();
        $fileName  = strtolower($extract[count($extract) - 1]);
        $reflector = new ReflectionClass(get_class($this));
        $formFile  = dirname(dirname(dirname(dirname($reflector->getFileName())))) . '/forms/filter_' . $fileName . '.xml';

        if (is_file($formFile)
            && ($xml = simplexml_load_file($formFile)) !== false
            && $form->load($xml)
        ) {
            $model        = $this->translationsModel;
            $filterValues = Factory::getApplication()->getUserStateFromRequest($model->getContext() . '.filter', 'filter', [], 'array');
            $inputFilter  = InputFilter::getInstance();

            foreach ($xml->xpath('fields[@name="filter"]/field') as $field) {
                $name = (string)$field->attributes()->name;

                if ((string)($field->attributes()->multiple ?? '') === 'true') {
                    $default = [];
                    $filter  = 'ARRAY';
                } elseif (str_ends_with(strtolower($name), 'id')) {
                    $default = 0;
                    $filter  = 'UINT';
                } else {
                    $default = '';
                    $filter  = 'STRING';
                }

                $value = $inputFilter->clean($filterValues[$name] ?? $default, $filter);
                $model->setState('filter.' . $name, $value);
                $form->setValue($name, 'filter', $value);
            }
        }
    }

    public function prepareListQuery(QueryInterface $query): void {}

    public function getComponentName(): string
    {
        return 'com_' . $this->extract()[0];
    }

    public function setForm(Form $form): TranslationExtensionAbstract
    {
        $this->form = $form;

        return $this;
    }

    public function getKey(): string
    {
        return 'id';
    }

    abstract public function getTitleField(): ?string;

    public function prepareDisplayData(array $translation, object $item): void
    {
        foreach ($translation as $field => $value) {
            if ((is_string($value) && trim($value) === '')
                || is_null($value)
                || !property_exists($item, $field)
            ) {
                continue;
            }

            if ($item->{$field} instanceof Registry) {
                $item->{$field}->merge(new Registry($value));
            } elseif (is_string($item->{$field}) && (str_contains($field, '.') || is_array($value)) || is_object($value)) {
                $item->{$field} = (new Registry($item->{$field}))->merge(new Registry($value))->toString();
            } elseif (is_array($item->{$field}) && is_array($value)) {
                $item->{$field} = ArrayHelper::mergeRecursive($item->{$field}, $value);
            } elseif (is_object($item->{$field}) && is_array($value)) {
                $item->{$field} = (object)ArrayHelper::mergeRecursive((array)$item->{$field}, $value);
            } else {
                // Force override value
                $item->{$field} = $value;
            }
        }
    }

    public function getTranslationFields($rootOnly = false): array
    {
        [$extension, $model] = $this->extract();
        $event = new RegisterTranslationFieldsEvent(
            [
                'fields'    => $this->registerTranslationFields(),
                'extension' => $extension,
                'model'     => $model,
                'subject'   => new stdClass(),
            ]
        );

        Factory::getApplication()
            ->getDispatcher()
            ->dispatch($event->getName(), $event);
        $fields = $event->getFields();

        if ($rootOnly) {
            $fields = array_map(function ($field) {
                return explode('.', $field)[0];
            }, $fields);
        }

        return array_unique($fields);
    }

    abstract public function registerTranslationFields(): array;

    public function prepareSaveData(string $context, string $langCode, object $item, array &$translation): void {}

    public function getDatabase(): DatabaseInterface
    {
        return Factory::getContainer()->get(DatabaseInterface::class);
    }

    public function getTitle(): string
    {
        $parts = explode('.', $this->getBaseContext());
        array_splice($parts, 1, -2);

        return Text::_(strtoupper((str_contains(get_class($this), 'Plugin\\System') ? 'COM' : 'PLG') . '_JLTRANSLATION_' . implode('_', $parts) . '_TITLE'));
    }

    public function getBaseContext(): string
    {
        static $baseContexts = [];
        $classname = get_class($this);

        if (!isset($baseContexts[$classname])) {
            $baseContexts[$classname] = str_replace(
                '\\',
                '.', strtolower(
                    preg_replace(
                        '/^.*\\\\Extension\\\\/',
                        '',
                        $classname
                    )
                )
            );
        }

        return $baseContexts[$classname];
    }

    public function getEditViewId(Input $input): array
    {
        return [
            $input->getCmd('view'),
            $input->getUint('id'),
        ];
    }
}