<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_joomlab
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\Joomlab\Form\Field;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Field\TextField;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Form\FormHelper;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\Plugin\System\Joomlab\Helper\LanguageHelper;
use Joomla\Plugin\System\Joomlab\Helper\StateHelper;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;
use SimpleXMLElement;
use Throwable;

class TranslationField extends FormField
{
    protected $type = 'Translation';

    protected string $filteredName = 'JLTranslationFilteredData';

    protected string $defaultLang = 'en-GB';

    protected bool $ignoreDefault = false;

    public function validate($value, $group = null, ?Registry $input = null)
    {
        if (!$this->hasTranslations()) {
            return parent::validate($value, $group, $input);
        }

        $postInput   = Factory::getApplication()->getInput()->post;
        $rawData     = $this->formControl ? $postInput->get($this->formControl, [], 'ARRAY') : $postInput->getArray();
        $inputData   = new Registry($rawData);
        $validData   = new Registry();
        $targetField = $this->getTargetField();
        $fieldType   = (string)$this->element['fieldType'];

        foreach (LanguageHelper::getLanguageCodes() as $langCode) {
            $postName  = $this->getTranslationGroup($langCode) . '.' . $this->fieldname;
            $fieldName = $langCode . '.' . ($this->group ? $this->group . '.' : '') . $this->fieldname;

            if ($this->defaultLang === $langCode) {
                $validData->set($fieldName, $value);
                continue;
            }

            $field                = clone $targetField;
            $fieldXml             = clone $this->element;
            $fieldXml['type']     = $fieldType;
            $fieldXml['required'] = 'false';
            $fieldValue           = $inputData->get($postName);
            $field->setup($fieldXml, $fieldValue, $group);
            $filteredValue = $field->filter($fieldValue);
            $validData->set($fieldName, $filteredValue);
            $validateResult = $field->validate($filteredValue, $group, $input);

            if (($validateResult instanceof Throwable) || (false === $validateResult)) {
                return $validateResult;
            }
        }

        $postInput->set(
            $this->filteredName,
            ArrayHelper::mergeRecursive(
                $postInput->get($this->filteredName, [], 'ARRAY'),
                $validData->toArray()
            )
        );

        return true;
    }

    protected function hasTranslations(): bool
    {
        return Multilanguage::isEnabled() && count(LanguageHelper::getLanguageCodes()) > 1;
    }

    protected function getInput()
    {
        static $scriptOptions = false;

        if (!$scriptOptions) {
            $scriptOptions = true;
            Factory::getApplication()
                ->getDocument()
                ->addScriptOptions('Joomlab', ['translation' => StateHelper::getInstance()->get('Joomlab.translation')], true);
        }

        $targetField = $this->getTargetField();

        if (!$this->hasTranslations()) {
            if (!$this->ignoreDefault && is_array($this->value)) {
                $this->value = $this->value[$this->defaultLang] ?? '';
            }

            $element         = clone $this->element;
            $element['type'] = (string)$element['fieldType'];
            $targetField->setup($element, $this->value, $this->group);

            return $targetField->__get('input');
        }

        $fields       = [];
        $stateContext = '';

        if ($this->form && ($pk = $this->form->getData()->get('id'))) {
            $stateContext = 'translation.' . $this->form->getName() . '.' . $pk;
        }

        $fieldValue = $this->value;

        if (is_object($fieldValue)) {
            $fieldValue = json_decode(json_encode($fieldValue), true);
        }

        $fieldType = (string)$this->element['fieldType'];

        foreach (LanguageHelper::getLanguageCodes() as $langCode) {
            $field            = clone $targetField;
            $fieldXml         = clone $this->element;
            $isDefault        = $this->defaultLang === $langCode;
            $fieldXml['type'] = $fieldType;
            $transform        = !$isDefault || !$this->ignoreDefault;

            if (!$isDefault) {
                $fieldXml['required'] = 'false';
            }

            $default = $fieldValue[$langCode] ?? '';

            if ($stateContext) {
                $stateName = $stateContext . '.' . $langCode . '.' . ($this->group ? $this->group . '.' : '') . $this->fieldname;
                $value     = StateHelper::getInstance()->get($stateName, $default);
            } else {
                $value = $default;
            }

            $field->setup($fieldXml, $value, $this->group);

            if ($transform) {
                $field->__set('group', $this->getTranslationGroup($langCode));
                $field->__set('name', $this->ignoreDefault ? $this->fieldname : $langCode);
            }

            $field->__set('id', (string)$fieldXml['id']);
            $tabField = ['input' => $field->__get('input'), 'lang' => $langCode];

            if ($isDefault) {
                array_unshift($fields, $tabField);
            } else {
                $fields[] = $tabField;
            }
        }

        return LayoutHelper::render(
            'form.field.translation',
            ['fields' => $fields],
            JPATH_PLUGINS . '/system/joomlab/tmpl'
        );
    }

    protected function getTargetField(): FormField
    {
        $fieldType  = ((string)$this->element['fieldType']) ?: 'text';
        $fieldClass = FormHelper::loadFieldClass($fieldType);

        if (!$fieldClass) {
            $fieldClass = TextField::class;
        }

        $form = $this->form ? clone $this->form : null;
        $form?->setFieldAttribute($this->fieldname, 'type', $fieldType, $this->group);

        return new $fieldClass($form);
    }

    public function setup(SimpleXMLElement $element, $value, $group = null)
    {
        $this->ignoreDefault = (string)$element['ignoreDefault'] === 'true';
        $this->defaultLang   = ComponentHelper::getParams('com_languages')->get('site', 'en-GB');

        return parent::setup($element, $value, $group);
    }

    protected function getTranslationGroup(string $langCode): string
    {
        if ($this->ignoreDefault) {
            return 'JLTranslation.' . $langCode . ($this->group ? '.' . $this->group : '');
        }

        return ($this->group ? $this->group . '.' : '.') . $this->fieldname;
    }

    public function filter($value, $group = null, ?Registry $input = null)
    {
        if ($this->ignoreDefault) {
            return parent::filter($value, $group, $input);
        }

        return $this->hasTranslations() ? (array)$value : [$this->defaultLang => $value];
    }
}