<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLTranslation\Administrator\Controller;

defined('_JEXEC') or die;

use Exception;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\Component\JLTranslation\Administrator\Model\TranslationModel;

class TranslationController extends BaseController
{
    public function toggle()
    {
        $this->checkToken();
        $language = $this->input->getString('language');
        $context  = $this->input->getString('context');
        $table    = $this->factory->createTable('Translation', 'Administrator');

        if ($context
            && $language
            && $table->load(['context' => $context, 'language' => $language])
        ) {
            $table->state = $table->state ? '0' : '1';

            if ($table->store()) {
                $this->app->enqueueMessage(
                    Text::sprintf(
                        $table->state
                            ? 'COM_JLTRANSLATION_LANGUAGE_ENABLED_MSG'
                            : 'COM_JLTRANSLATION_LANGUAGE_DISABLED_MSG',
                        $language
                    )
                );
            }
        }

        $redirect = 'index.php';

        if ($return = $this->input->getBase64('return')) {
            $return = base64_decode($return);

            if (Uri::isInternal($return)) {
                $redirect = $return;
            }
        }

        $this->app->redirect($redirect);
    }

    public function cancel($key = null): void
    {
        $this->app->redirect($this->getRedirectListUrl());
    }

    protected function getRedirectListUrl()
    {
        $filter = $this->input->get('filter', [], 'ARRAY');
        $url    = 'index.php?option=com_jltranslation&view=translations';

        if ($table = $filter['table'] ?? null) {
            $url .= '&filter[table]=' . $table;
        }

        return Route::_($url, false);
    }

    public function apply()
    {
        $this->saveTranslation();
        $this->setRedirect($this->getRedirectItemUrl(), Text::_('COM_JLTRANSLATION_TRANSLATION_SAVED'));
    }

    protected function saveTranslation()
    {
        $this->checkToken();

        /**
         * @var TranslationModel $model
         * @var Form             $translationForm
         */
        $data  = $this->input->post->get('JLTranslation', [], 'ARRAY');
        $model = $this->getModel('Translation', 'Administrator');
        [, $translationForm] = $model->getForms();
        $return = $translationForm->process($data);

        // Check for an error.
        if ($return instanceof Exception) {
            $this->app->enqueueMessage($return->getMessage(), CMSApplicationInterface::MSG_ERROR);
            $this->app->redirect($this->getRedirectItemUrl());
        }

        // Check the validation results.
        if ($return === false) {
            // Get the validation messages from the form.
            foreach ($translationForm->getErrors() as $message) {
                $this->app->enqueueMessage($message, CMSApplicationInterface::MSG_ERROR);
            }

            $this->app->redirect($this->getRedirectItemUrl());
        }

        $validData = $return;
        $model->save($validData);
    }

    protected function getRedirectItemUrl()
    {
        $filter = $this->input->get('filter', [], 'ARRAY');
        $url    = 'index.php?option=com_jltranslation&view=translation';

        if ($table = $filter['table'] ?? null) {
            $url .= '&filter[table]=' . $table;
        }

        if ($language = $filter['language'] ?? null) {
            $url .= '&filter[language]=' . $language;
        }

        if ($tmpl = $this->input->get('tmpl')) {
            $url .= '&tmpl=' . $tmpl;
        }

        $url .= '&layout=edit&id=' . $this->input->get('id', 0, 'UINT');

        return Route::_($url, false);
    }

    public function save()
    {
        $this->saveTranslation();
        $this->setRedirect($this->getRedirectListUrl(), Text::_('COM_JLTRANSLATION_TRANSLATION_SAVED'));
    }
}
