<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLTranslation\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;
use Joomla\Database\QueryInterface;
use Joomla\Plugin\System\JLTranslation\Driver\DatabaseDriver;
use Joomla\Plugin\System\JLTranslation\Extension\TranslationExtensionAbstract;
use Joomla\Plugin\System\JLTranslation\Helper\TranslationHelper;
use RuntimeException;

class TranslationsModel extends ListModel
{
    public function __construct($config = [])
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = ['id', 'title'];
        }

        parent::__construct($config);
    }

    public function getItems()
    {
        $table = $this->getState('filter.table') ?: 'content';

        if (!$table || !TranslationHelper::getClassByTable($table)) {
            return [];
        }

        return parent::getItems();
    }

    public function getContext(): string
    {
        return $this->context;
    }

    protected function populateState($ordering = 'id', $direction = 'asc')
    {
        // List state information.
        parent::populateState($ordering, $direction);
    }

    protected function getStoreId($id = ''): string
    {
        // Compile the store id.
        foreach ($this->getFilterForm()->getGroup('filter') as $field) {
            $name  = $field->getAttribute('name');
            $value = $this->getState('filter.' . $name);

            if (is_array($value) || is_object($value)) {
                $value = serialize($value);
            }

            $id .= ':' . $value;
        }

        return parent::getStoreId($id);
    }

    public function getFilterForm($data = [], $loadData = true)
    {
        static $filterForms = [];
        $table = $this->getState('filter.table') ?: 'content';

        if (!isset($filterForms[$table])) {
            if (($filterForms[$table] = parent::getFilterForm($data, $loadData))
                && $table
                && ($class = TranslationHelper::getClassByTable($table))
            ) {
                $class->setTranslationsModel($this)
                    ->prepareFilterForm($filterForms[$table]);
            }
        }

        return $filterForms[$table];
    }

    protected function getListQuery(): QueryInterface
    {
        $table = $this->getState('filter.table') ?: 'content';

        if (!$table
            || !($class = TranslationHelper::getClassByTable($table))
            || !($titleField = $class->getTitleField())
        ) {
            throw new RuntimeException('Invalid reference table');
        }

        /**
         * @var TranslationExtensionAbstract $class
         * @var DatabaseDriver               $db
         */
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);
        $idKey = $class->getKey();
        $query->select(
            $this->getState(
                'list.select',
                $db->quoteName('a.' . $idKey, 'id')
                . ', ' . $db->quoteName('a.' . $titleField, 'title')
            )
        )
            ->from($db->quoteName('#__' . $table, 'a'));

        // Filter by search in title.
        $search = $this->getState('filter.search');

        if (!empty($search)) {
            if (stripos($search, 'id:') === 0) {
                $search = (int)substr($search, 3);
                $query->where($db->quoteName('a.' . $idKey) . ' = :search')
                    ->bind(':search', $search, ParameterType::INTEGER);
            } else {
                $search = '%' . trim($search) . '%';
                $query->where($db->quoteName('a.' . $titleField) . ' LIKE :search')
                    ->bind(':search', $search);
            }
        }

        // Add the list ordering clause.
        $orderCol  = $this->state->get('list.ordering', 'id');
        $orderDirn = $this->state->get('list.direction', 'ASC');
        $query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));
        $class->setTranslationsModel($this)
            ->prepareListQuery($query);

        return $query;
    }
}
