<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_joomlab
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\JLAdmin\Extension;

defined('_JEXEC') or die;

use Joomla\CMS\Event\Model\PrepareFormEvent;
use Joomla\CMS\Event\User\LoginEvent;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\User\UserFactoryAwareTrait;
use Joomla\Component\JLAdmin\Administrator\Helper\ComponentHelper;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Event\SubscriberInterface;

if (JLADMIN_PRO) {
    PluginHelper::importPlugin('jladmin', 'pro');
}

final class JLAdmin extends CMSPlugin implements SubscriberInterface
{
    use DatabaseAwareTrait;
    use UserFactoryAwareTrait;

    private static bool $isAttempts = false;

    public static function getSubscribedEvents(): array
    {
        return [
            'onAfterInitialise'    => 'onAfterInitialise',
            'onContentPrepareForm' => 'onContentPrepareForm',
            'onUserLogin'          => 'onUserLogin',
        ];
    }

    public function onAfterInitialise(): void
    {
        $this->checkAdminAlias();
    }

    private function checkAdminAlias()
    {
        $app        = $this->getApplication();
        $input      = $app->getInput();
        $adminAlias = trim(ComponentHelper::getConfig('adminPathAlias', ''), '/\\');

        if ($adminAlias && !$app->get('sef_rewrite')) {
            if ($app->isClient('administrator')) {
                $app->enqueueMessage('JLAdmin – The admin path alias feature is only effective when your website is configured to use URL Rewriting.', 'warning');
            }

            return;
        }

        $stateAliasKey  = 'plg_system_jladmin.adminAlias';
        $hasAdminAlias  = $adminAlias && $adminAlias !== 'administrator';
        $hashAdminAlias = md5($adminAlias);
        $isAdminRef     = $hasAdminAlias && $input->exists($hashAdminAlias);
        $isAdmin        = $app->isClient('administrator');
        $task           = $input->getCmd('task', '');

        if ($isAdminRef
            || ($isAdmin
                && $input->getCmd('option') === 'com_config'
                && $input->getCmd('component') === 'com_jladmin'
                && $input->getMethod() !== 'GET'
                && (str_ends_with($task, 'apply') || str_ends_with($task, 'save'))
            )
        ) {
            $app->setUserState($stateAliasKey, true);
        }

        if (true === $app->getUserState($stateAliasKey)) {
            return;
        }

        $rootUri = Uri::root(true) . '/';

        if ($isAdmin && $hasAdminAlias) {
            // Path alias is incorrect redirect to the site page
            $app->redirect($rootUri);
        }

        $currentPath  = trim(Uri::getInstance()->getPath(), '/');
        $currentAlias = trim($rootUri . $adminAlias, '/');

        if (!$isAdmin
            && $hasAdminAlias
            && $currentPath === $currentAlias
        ) {
            // Redirect to right administrator path
            $app->redirect($rootUri . 'administrator?' . $hashAdminAlias);
        }
    }

    public function onContentPrepareForm(PrepareFormEvent $event)
    {
        $this->loadRegistrationFormReCaptcha($event);
        $form = $event->getForm();

        if ('com_config.component' !== $form->getName()
            || 'com_jladmin' !== $this->getApplication()->getInput()->get('component')
            || JLADMIN_PRO
        ) {
            return;
        }

        $this->loadLanguage('com_jladmin.sys', JPATH_ADMINISTRATOR . '/components/com_jladmin');
        $names = [
            'backupCloudStorage',
            'googleDriveCredentials',
            'backupProfilesForCLI',
            'maxFilesInCloud',
            'useRateLimiter',
            'enableMailTracker',
            'backupRemoveAfterDays',
            'blockUserAfterFailedTimes',
            'blockUserWarningMessage',
            'blockUserEmailMessage',
        ];

        foreach ($names as $fieldName) {
            $field = $form->getField($fieldName);
            $label = $field->getAttribute('label');
            $form->setFieldAttribute($fieldName, 'translateLabel', 'false');
            $form->setFieldAttribute($fieldName, 'label', Text::_($label) . ' <span class="badge bg-danger">PRO</span>');
        }
    }

    private function loadRegistrationFormReCaptcha(PrepareFormEvent $event)
    {
        $form   = $event->getForm();
        $app    = $this->getApplication();
        $config = ComponentHelper::getConfig();

        if ('com_users.registration' !== $form->getName()
            || !$app->isClient('site')
            || !($useReCaptcha = $config->get('useReCaptcha', ''))
        ) {
            return;
        }

        $xml      = simplexml_load_string('<form></form>');
        $fieldSet = $xml->addChild('fieldset');
        $fieldSet->addAttribute('name', 'default');
        $reCaptchaField = $fieldSet->addChild('field');
        $reCaptchaField->addAttribute('name', 'reCaptcha');
        $reCaptchaField->addAttribute('type', 'reCaptcha');
        $reCaptchaField->addAttribute('addfieldprefix', 'Joomla\\Plugin\\System\\Joomlab\\Form\\Field');
        $reCaptchaField->addAttribute('version', $useReCaptcha);
        $reCaptchaField->addAttribute('siteKey', $config->get('reCaptchaSiteKey' . $useReCaptcha));
        $reCaptchaField->addAttribute('secretKey', $config->get('reCaptchaSecretKey' . $useReCaptcha));
        $reCaptchaField->addAttribute('hiddenLabel', 'true');
        $form->load($xml);
    }

    public function onUserLogin(LoginEvent $event): void
    {
        $app    = $this->getApplication();
        $config = ComponentHelper::getConfig();

        if ($app->isClient('site')) {
            $user     = $event->getAuthenticationResponse();
            $instance = $this->getUserFactory()->loadUserByUsername($user['username']);

            if ($config->get('preventAdminLoginOnSite', 0) && $instance->authorise('core.admin')) {
                $event->addResult(false);
                $event->stopPropagation();
                $app->enqueueMessage($app->getLanguage()->_('JLIB_LOGIN_DENIED'), 'error');
                $app->logout();
            }

            return;
        }

        $loginTimes = (int)$config->get('loginTimes', 1);

        if ($loginTimes <= 1 || !$app->isClient('administrator')) {
            return;
        }

        $tryKey   = 'plg_system_jladmin.loginAttempts';
        $attempts = (int)$app->getUserState($tryKey, 1);

        if ($attempts < $loginTimes) {
            self::$isAttempts = true;
            $app->setUserState($tryKey, $attempts + 1);
            $event->addResult(false);
            $event->stopPropagation();
            $time = $loginTimes - $attempts;
            $app->enqueueMessage($config->get('loginTimesMsg') ?: 'Great! try to ' . $time . ' more time' . ($time > 1 ? 's' : ''), 'warning');
        } else {
            $app->setUserState($tryKey, null);
        }
    }
    
    public static function isAttempts(): bool
    {
        return self::$isAttempts;
    }
}