<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLAdmin\Administrator\Model;

defined('_JEXEC') or die;

use FilesystemIterator;
use Joomla\CMS\Factory;
use Joomla\Component\JLAdmin\Administrator\Helper\ComponentHelper;
use Joomla\Database\ParameterType;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Path;
use Joomla\Plugin\System\Joomlab\Helper\ZipHelper;
use Joomla\Plugin\System\Joomlab\MVC\Model\AdminModel;
use Joomla\Utilities\ArrayHelper;
use RecursiveCallbackFilterIterator;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use RuntimeException;
use SplFileInfo;
use Throwable;

class BackupModel extends AdminModel
{
    public function backup(?int $profileId = null, ?callable $callback = null): void
    {
        $db        = $this->getDatabase();
        $profileId = $profileId ?? 0;

        if (!$profileId) {
            $query     = $db->getQuery(true)
                ->select($db->quoteName('id'))
                ->from($db->quoteName('#__joomlab_backup_profiles'))
                ->where($db->quoteName('isDefault') . ' = 1');
            $profileId = $db->setQuery($query)->loadResult();
        }

        $query = $db->getQuery(true)
            ->select($db->quoteName(['id', 'params']))
            ->from($db->quoteName('#__joomlab_backup_profiles'))
            ->where($db->quoteName('id') . ' = :id')
            ->bind(':id', $profileId, ParameterType::INTEGER);

        if (!($row = $db->setQuery($query)->loadObject())) {
            throw new RuntimeException('Invalid backup profile ID');
        }

        $params = json_decode($row->params ?: '{}', true);

        if (!empty($params['excludePaths'])) {
            $excludePaths = preg_split('/\r\n|\n/', trim($params['excludePaths']), -1, PREG_SPLIT_NO_EMPTY);
        } else {
            $excludePaths = [];
        }

        $excludePaths[] = '/administrator/components/com_jladmin/backups';

        if (!$callback) {
            $callback = function ($step, $data) {};
        }

        call_user_func_array(
            $callback,
            ['startingBackup', []]
        );

        $backupType = $params['backupType'] ?? 'fullSite';
        $app        = Factory::getApplication();
        $user       = $app->getIdentity();
        $date       = Factory::getDate('now', $user?->getTimezone() ?? $app->get('offset', 'UTC'));
        $backupName = str_ireplace(
            [
                '[SITE]',
                '[DATE]',
                '[TIME]',
                '[TZ]',
            ],
            [
                $app->get('sitename'),
                $date->format('Y-m-d', true),
                $date->format('H:i:s', true),
                $user?->getTimezone()->getName() ?? $app->get('offset', 'UTC'),
            ],
            ($params['backupName'] ?? '') ?: 'site-[SITE]-[DATE]T[TIME]-[TZ]'
        );

        $backupName  = uniqid() . '__' . preg_replace('/[^a-z0-9_\-:]/i', '_', $backupName) . '.zip';
        $archiveFile = JPATH_ADMINISTRATOR . '/components/com_jladmin/backups/' . $backupName;
        $rootOffset  = strlen(JPATH_ROOT) + 1;
        $zip         = new ZipHelper($archiveFile);
        $isFullSite  = 'fullSite' === $backupType;

        if ($isFullSite) {
            $zip
                ->addFile([
                    'name' => 'installation/index.php',
                    'file' => JPATH_ADMINISTRATOR . '/components/com_jladmin/layouts/restore-index.php.txt',
                ]);
        }

        if (in_array($backupType, ['fullSite', 'filesSite'])) {
            // Collect files
            $iterator     = new RecursiveDirectoryIterator(JPATH_ROOT, FilesystemIterator::SKIP_DOTS);
            $filter       = new RecursiveCallbackFilterIterator($iterator, function ($current) use ($excludePaths, $rootOffset) {
                /** @var SplFileInfo $current */
                $filePath = '/' . rtrim(Path::clean(substr($current->getRealPath(), $rootOffset), '/'), '/');
                $fileName = basename($filePath);

                foreach ($excludePaths as $excludePath) {
                    $excludePath = rtrim(Path::clean($excludePath, '/'), '/');

                    if (($excludePath[0] === '/' && str_starts_with($filePath, $excludePath))
                        || ($excludePath[0] !== '/' && $fileName === $excludePath)
                    ) {
                        return false;
                    }
                }

                return true;
            });
            $files        = new RecursiveIteratorIterator($filter);
            $totalFiles   = iterator_count($files);
            $scannedFiles = 0;
            $batchSize    = 100;

            /** @var SplFileInfo $file */
            foreach ($files as $file) {
                if (!$file->isFile() || !$file->isReadable()) {
                    continue;
                }

                $scannedFiles++;
                $filePath     = $file->getRealPath();
                $relativePath = substr($filePath, $rootOffset); // Remove base folder path
                $zip->addFile([
                    'name' => $relativePath,
                    'file' => $file,
                    'time' => $file->getATime(),
                ]);

                // Send update every $batchSize files
                if ($scannedFiles % $batchSize === 0 || $scannedFiles === $totalFiles) {
                    call_user_func_array(
                        $callback,
                        [
                            'collectingFiles',
                            [
                                'file'         => $file,
                                'totalFiles'   => $totalFiles,
                                'scannedFiles' => $scannedFiles,
                            ],
                        ]
                    );
                }
            }

            call_user_func_array(
                $callback,
                [
                    'collectFilesDone',
                    [
                        'totalFiles'   => $totalFiles,
                        'scannedFiles' => $scannedFiles,
                    ],
                ]
            );
        }

        if (in_array($backupType, ['fullSite', 'dbSite'])) {
            call_user_func_array(
                $callback,
                ['startingBackupDatabase', []]
            );

            $prefix       = $db->getPrefix();
            $useMysqlDump = ComponentHelper::getConfig('backupUseMySqlDump', 1)
                && function_exists('shell_exec')
                && shell_exec('which mysqldump');

            if ($useMysqlDump) {
                $tmpSQLFile = JPATH_ROOT . '/tmp/mysqldump-' . uniqid() . '.sql';
                $host       = $app->get('host');
                $port       = 3306;

                if (str_contains($host, ':')) {
                    [$host, $port] = explode(':', $host, 2);
                }

                $username = $app->get('user');
                $password = $app->get('password');
                $database = $app->get('db');
                // Command to run mysqldump
                $command = "mysqldump -h $host -P $port -u $username -p$password $database > $tmpSQLFile 2>/dev/null";

                // Execute the command
                shell_exec($command);

                if (is_file($tmpSQLFile)) {
                    $sqlDump = str_replace($prefix, '#__', file_get_contents($tmpSQLFile));

                    // Try to remove $tmpSQLFile
                    try {
                        File::delete($tmpSQLFile);
                    } catch (Throwable $e) {
                    }
                }
            }

            if (!isset($sqlDump)) {
                // Create SQL file
                $sqlData       = [];
                $listTables    = $db->getTableList();
                $totalTables   = count($listTables);
                $scannedTables = 0;

                foreach ($listTables as $table) {
                    $tableNonPrefix = str_replace($prefix, '#__', $table);
                    [, $createData] = $db
                        ->setQuery('SHOW CREATE TABLE ' . $db->quoteName($table))
                        ->loadRow();

                    $sqlData[] = 'DROP TABLE IF EXISTS ' . $db->quoteName($tableNonPrefix) . ';';
                    $sqlData[] = str_replace($prefix, '#__', $createData) . ';';
                    $query     = $db->getQuery(true)
                        ->select('*')
                        ->from($db->quoteName($tableNonPrefix));

                    if ($rows = $db->setQuery($query)->loadObjectList()) {
                        $sqlData[] = 'LOCK TABLES ' . $db->quoteName($tableNonPrefix) . ' WRITE;';
                        $insertSql = 'INSERT INTO ' . $db->quoteName($tableNonPrefix) . ' (';

                        foreach (array_keys((array)$rows[0]) as $column) {
                            $insertSql .= $db->quoteName($column) . ',';
                        }

                        $insertSql = rtrim($insertSql, ',') . ') VALUES ';

                        foreach ($rows as $row) {
                            $values = '(';

                            foreach ($row as $value) {
                                if (is_null($value)) {
                                    $values .= 'NULL,';
                                } else {
                                    $values .= $db->quote($value) . ',';
                                }
                            }

                            $insertSql .= rtrim($values, ',') . '),';
                        }

                        $sqlData[] = rtrim($insertSql, ',') . ';';
                        $sqlData[] = 'UNLOCK TABLES;' . PHP_EOL;
                    }

                    call_user_func_array(
                        $callback,
                        [
                            'backupDataTableDone',
                            [
                                'table'         => $tableNonPrefix,
                                'totalTables'   => $totalTables,
                                'scannedTables' => ++$scannedTables,
                            ],
                        ]
                    );
                }

                $sqlDump = implode(PHP_EOL, $sqlData);
            }

            $zip->addFile([
                'name' => $isFullSite ? 'installation/data.sql' : 'data.sql',
                'data' => $sqlDump,
                'time' => Factory::getDate()->toUnix(),
            ]);

            call_user_func_array(
                $callback,
                [
                    'backupDatabaseDone',
                    [],
                ]
            );
        }

        call_user_func_array(
            $callback,
            ['startingCompressBackupFile', []]
        );

        $zip->close();
        call_user_func_array(
            $callback,
            [
                'compressBackupFileDone',
                [
                    'archiveFile' => $archiveFile,
                ],
            ]
        );

        // Store the backup to the database
        $backupTable = $this->getMVCFactory()
            ->createTable('Backup', 'Administrator');
        $backupTable->bind([
            'profileId'   => (int)$profileId,
            'clientId'    => $app->isClient('cli') ? 2 : 1,
            'backupFile'  => $backupName,
            'createdDate' => $date->toSql(),
            'createdBy'   => $user?->id ?? 0,
        ]);

        if (!$backupTable->store()) {
            throw new RuntimeException($backupTable->getError());
        }

        if (JLADMIN_PRO) {
            call_user_func_array([
                $app->bootPlugin('pro', 'jladmin'),
                'onBackupSuccess',
            ], [
                [
                    'backupTable' => $backupTable,
                    'archiveFile' => $archiveFile,
                    'callback'    => $callback,
                ],
            ]);
        }
    }
    public function delete(&$pks)
    {
        $pks         = ArrayHelper::toInteger((array) $pks);
        $db          = $this->getDatabase();
        $query       = $db->createQuery()
            ->select($db->quoteName(['id', 'backupFile']))
            ->from($db->quoteName('#__joomlab_backups'))
            ->whereIn($db->quoteName('id'), $pks);
        $backupFiles = $db->setQuery($query)->loadObjectList('id');

        if ($result = parent::delete($pks)) {
            foreach ($pks as $pk) {
                if (isset($backupFiles[$pk])) {
                    $file = JPATH_ADMINISTRATOR . '/components/com_jladmin/backups/' . $backupFiles[$pk]->backupFile;

                    if (is_file($file)) {
                        try {
                            File::delete($file);
                        } catch (Throwable $e) {
                            Factory::getApplication()->enqueueMessage($e->getMessage(), 'warning');
                        }
                    }
                }
            }
        }

        return $result;
    }
}
