<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_joomlab
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\Joomlab\Form\Field;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Form\Field\HiddenField;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Language\Text;
use Joomla\Plugin\System\Joomlab\Helper\AssetHelper;
use Joomla\Plugin\System\Joomlab\Helper\UtilityHelper;
use Joomla\Registry\Registry;
use RuntimeException;

class ReCaptchaField extends HiddenField
{
    protected $type = 'ReCaptcha';

    public function validate($value, $group = null, ?Registry $input = null)
    {
        $isValidCaptcha = false;
        $version        = (int)$this->getAttribute('version');

        if ($recaptchaToken = $value ?: Factory::getApplication()->getInput()->getString('g-recaptcha-response')) {
            $recaptchaResponse = HttpFactory::getHttp()
                ->post('https://www.google.com/recaptcha/api/siteverify', [
                    'secret'   => $this->getAttribute('secretKey'),
                    'response' => $recaptchaToken,
                    'remoteip' => UtilityHelper::getClientIP(),
                ]);
            $responseJson      = json_decode($recaptchaResponse->body, true);
            $isValidCaptcha    = !empty($responseJson['success']);

            if ($isValidCaptcha && $version === 3) {
                $score          = (float)($responseJson['score'] ?? 0.0);
                $isValidCaptcha = $score >= (float)$this->getAttribute('score', '0.5');
            }
        }

        if (!$isValidCaptcha) {
            return new RuntimeException(Text::_('PLG_SYSTEM_JOOMLAB_INVALID_RECAPTCHA_MSG'));
        }
    }

    protected function getInput()
    {
        $input   = parent::getInput();
        $siteKey = $this->getAttribute('siteKey');
        $version = (int)$this->getAttribute('version');
        Factory::getApplication()
            ->getDocument()
            ->addScriptOptions('plg_system_joomlab', [
                'reCaptchaSiteKey' => $siteKey,
                'reCaptchaVersion' => $version,
            ]);

        $wa  = AssetHelper::getWebAssetManager();
        $uri = 'https://www.google.com/recaptcha/api.js' . ($version === 3 ? '?render=' . $siteKey : '');

        if ($version === 2) {
            $wa->addInlineScript(<<<JAVASCRIPT
function onJLRecaptcha2Callback(token) {
    const {reCaptchaSiteKey, reCaptchaVersion} = Joomla.getOptions('plg_system_joomlab');
    const input = document.getElementById('$this->id');

    if (window.grecaptcha
        && reCaptchaSiteKey
        && reCaptchaVersion === 2) {
        input.value = token;
    }
}

function onJLRecaptcha2ExpiredCallback() {
    if (window.grecaptcha) {
        const input = document.getElementById('$this->id');
        input.value = '';
        grecaptcha.reset();
    }
}

JAVASCRIPT
            );

            $input .= '<div class="g-recaptcha" data-sitekey="' . $siteKey . '" data-callback="onJLRecaptcha2Callback" data-expired-callback="onJLRecaptcha2ExpiredCallback"></div>';
        } elseif ($version === 3) {
            $wa->addInlineScript(<<<JAVASCRIPT
document.addEventListener('DOMContentLoaded', function () {
    const {reCaptchaSiteKey, reCaptchaVersion} = Joomla.getOptions('plg_system_joomlab');
    const input = document.getElementById('$this->id');
    const form = input.closest('form');
        
    if (!window.grecaptcha
        || !reCaptchaSiteKey
        || reCaptchaVersion !== 3
        || !form
    ) {
        return;
    }
    
    grecaptcha.ready(() => {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            grecaptcha.execute(reCaptchaSiteKey, {action: 'submit'}).then((token) => {
                input.value = token;
                form.submit(); 
            }).catch(console.error); 
        });
        
    });
});
JAVASCRIPT
            );
        }

        $wa->registerAndUseScript('plg_system_joomlab.recaptcha', $uri, [], ['defer' => true]);

        return $input;
    }
}