<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_joomlab
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\Joomlab\Helper;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use RuntimeException;
use SplFileInfo;

class FileSystemHelper
{
    public static function stream(string $path, bool $removeAfter = false)
    {
        if (!file_exists($path)) {
            throw new RuntimeException('Resource not exists: ' . $path);
        }

        static::increaseMemoryLimit();

        $app = Factory::getApplication();

        if (is_file($path)) {
            $mime = mime_content_type($path);

            if (false === $mime) {
                throw new RuntimeException('Unknown file type: ' . $path);
            }

            $filesize = filesize($path);

            if (str_starts_with($mime, 'image/')) {
                $app
                    ->setHeader('Content-Type', $mime, true)
                    ->sendHeaders();
                readfile($path);
            } elseif (str_starts_with($mime, 'video/')) {
                $app
                    ->setHeader('Content-Type', $mime, true)
                    ->setHeader('status', '206 Partial Content', true)
                    ->setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0', true)
                    ->setHeader('Pragma', 'no-cache', true);

                // Check if the browser requests a specific range (seek support)
                if ($range = $app->getInput()->server->getString('HTTP_RANGE')) {
                    [, $range] = explode('=', $range, 2);
                    [$start, $end] = explode('-', $range);

                    $start = intval($start);
                    $end   = $end === '' ? ($filesize - 1) : intval($end);
                    $app
                        ->setHeader('Accept-Ranges', 'bytes', true)
                        ->setHeader('Content-Range', "bytes $start-$end/$filesize", true)
                        ->sendHeaders();
                    $fp = fopen($path, 'rb');
                    fseek($fp, $start);
                    echo fread($fp, $end - $start + 1);
                    fclose($fp);
                } else {
                    $app
                        ->setHeader('Content-Length', $filesize, true)
                        ->sendHeaders();
                    readfile($path);
                }
            } else {
                $filename = basename($path);
                $headers  = [
                    'Content-Type'              => $mime === 'application/zip' ? $mime : 'application/octet-stream',
                    'Content-Length'            => $filesize,
                    'Content-Disposition'       => 'attachment; filename="' . htmlspecialchars(str_replace('"', '', $filename)) . '"',
                    'Cache-Control'             => 'no-store, no-cache',
                    'Pragma'                    => 'no-cache',
                    'Accept-Ranges'             => 'bytes',
                    'Content-Transfer-Encoding' => 'binary',
                    'Connection'                => 'close',
                ];

                foreach ($headers as $name => $value) {
                    $app->setHeader($name, $value);
                }

                $app->sendHeaders();
                $blockSize = 1048576; // 1048576 = 1M chunks
                $handle    = fopen($path, 'r');

                if ($handle !== false) {
                    while (!feof($handle)) {
                        echo fread($handle, $blockSize);
                        ob_flush();
                        flush();
                    }

                    fclose($handle);
                }
            }

            if ($removeAfter) {
                File::delete($path);
            }

            $app->close();
        }

        // The path is a folder, compress it as a ZIP file and stream it
        $archive = JPATH_ROOT . '/tmp/' . basename($path) . '.zip';
        static::compress($archive, $path);

        return static::stream($archive, true);
    }

    public static function increaseMemoryLimit()
    {
        static $done = false;

        if ($done) {
            return;
        }

        $done = true;
        if (function_exists('ini_set') && function_exists('ini_get')) {
            if (ini_get('zlib.output_compression')) {
                ini_set('zlib.output_compression', 'Off');
            }

            if (function_exists('set_time_limit') && !ini_get('safe_mode')) {
                set_time_limit(0);
            }

            ini_set('memory_limit', '-1');
        }
    }

    public static function getExt(string $file): string
    {
        preg_match('/\.([a-z0-9]+)(\?.*)?$/i', ltrim($file, '.'), $matches);

        return $matches[1] ?? '';
    }

    public static function isEmptyFolder(string $path): bool
    {
        return is_dir($path) && !(glob($path . '/*') || glob($path . '/.*'));
    }

    public static function stripExt(string $file): string
    {
        $dot = str_starts_with($file, '.');

        return ($dot ? '.' : '') . preg_replace('#\.[^.]*$#', '', ltrim($file, '.'));
    }

    public static function getUploadMaxSizes($toMB = false): float
    {
        static $uploadMaxSizes = null;

        if (null === $uploadMaxSizes) {
            $uploadMaxSizes = static::parseSize(ini_get('upload_max_filesize'));
            $postMaxSizes   = static::parseSize(ini_get('post_max_size'));

            if ($postMaxSizes > 0 && $uploadMaxSizes < $postMaxSizes) {
                $uploadMaxSizes = $postMaxSizes;
            }
        }

        return $toMB ? static::toMB($uploadMaxSizes) : $uploadMaxSizes;
    }

    private static function parseSize($size): float
    {
        $unit   = preg_replace('/[^bkmgtpezy]/i', '', $size);
        $size   = preg_replace('/[^0-9.]/', '', $size);
        $return = round($size);

        if ($unit) {
            $return = round($size * pow(1024, stripos('bkmgtpezy', $unit[0])));
        }

        return $return;
    }

    public static function toMB(int|float $bytes, int $precision = 2): float
    {
        return round($bytes / (1024 * 1024), $precision);
    }

    public static function getFileSize(string $path, bool $toMB = false): int|float|null
    {
        if (is_file($path)) {
            $size = filesize($path);

            return $toMB ? static::toMB($size) : $size;
        }

        return null;
    }

    public static function readFile(string|SplFileInfo $file, ?int $chunkSize = 1024 * 1024): string
    {
        // $chunkSize = 1024 * 1024 -> 1MB = 1,048,576 bytes
        $data = '';

        if ($file instanceof SplFileInfo) {
            if (!$file->isFile() || !$file->isReadable()) {
                throw new RuntimeException('File is not readable: ' . $file->getRealPath());
            }

            if (null === $chunkSize) {
                return file_get_contents($file->getRealPath());
            }

            $file = $file->openFile();

            while (!$file->eof()) {
                $data .= $file->fread($chunkSize);
            }
        } else {
            if (!is_file($file)) {
                throw new RuntimeException('File does not exist: ' . $file);
            }

            if (null === $chunkSize) {
                return file_get_contents($file);
            }

            if (false === ($fp = fopen($file, 'r'))) {
                throw new RuntimeException('Could not open the file: ' . $file);
            }

            while (!feof($fp)) {
                $data .= fread($fp, $chunkSize);
            }
        }

        return $data;
    }

    public static function cleanFolder(string|array $folderPath, bool $keepIndex = true): void
    {
        foreach (array_unique(is_array($folderPath) ? $folderPath : [$folderPath]) as $path) {
            if (is_dir($path)) {
                foreach (Folder::folders($path, '.', true, true, []) as $folder) {
                    Folder::delete($folder);
                }

                foreach (Folder::files($path, '.', true, true, []) as $file) {
                    File::delete($file);
                }

                if ($keepIndex) {
                    File::write($path . '/index.html', '<!DOCTYPE html><title></title>');
                }
            }
        }
    }
}