<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\Joomlab\Helper;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Helper\ModuleHelper;
use Joomla\CMS\Installer\Installer;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Database\DatabaseInterface;
use Joomla\Filesystem\Folder;
use SimpleXMLElement;
use Throwable;

class InstallerHelper
{
    public static function installExtensions(string $source)
    {
        // Plugins
        if (is_dir($source . '/plugins')) {
            foreach (Folder::folders($source . '/plugins') as $group) {
                foreach (Folder::folders($source . '/plugins/' . $group) as $name) {
                    $pluginPath   = $source . '/plugins/' . $group . '/' . $name;
                    $isSystemCore = $group === 'system' && $name === 'joomlab';

                    if ($isSystemCore) {
                        try {
                            $upVersion      = (string)(new SimpleXMLElement(file_get_contents($pluginPath . '/joomlab.xml')))->version;
                            $currentVersion = null;

                            if (PluginHelper::isEnabled('system', 'joomlab')) {
                                $currentVersion = (string)(new SimpleXMLElement(file_get_contents(JPATH_PLUGINS . '/system/joomlab/joomlab.xml')))->version;
                            }

                            if ($currentVersion && version_compare($currentVersion, $upVersion, 'gt')) {
                                continue;
                            }
                        } catch (Throwable $e) {
                            Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
                            continue;
                        }
                    }

                    $installer = new Installer();
                    $installer->setPath('source', $pluginPath);
                    $route      = PluginHelper::isEnabled($group, $name) ? 'update' : 'install';
                    $newAdapter = $installer->setupInstall($route, true);

                    if (is_object($newAdapter) && $newAdapter->{$route}()) {
                        $db    = Factory::getContainer()->get(DatabaseInterface::class);
                        $query = $db->getQuery(true)
                            ->update($db->quoteName('#__extensions'))
                            ->set($db->quoteName('enabled') . ' = 1')
                            ->set($db->quoteName('protected') . ' = ' . ($isSystemCore ? '1' : '0'))
                            ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
                            ->where($db->quoteName('folder') . ' = ' . $db->quote($group))
                            ->where($db->quoteName('element') . ' = ' . $db->quote($name));
                        $db->setQuery($query)->execute();
                    }
                }
            }
        }

        // Modules
        if (is_dir($source . '/modules')) {
            foreach (Folder::folders($source . '/modules', '^mod_', false, true) as $modulePath) {
                $installer = new Installer();
                $installer->setPath('source', $modulePath);
                $route      = ModuleHelper::isEnabled(basename($modulePath)) ? 'update' : 'install';
                $newAdapter = $installer->setupInstall($route, true);

                if (is_object($newAdapter)) {
                    $newAdapter->{$route}();
                }
            }
        }
    }

    public static function uninstallExtensions(array $sources)
    {
        $db = Factory::getContainer()->get(DatabaseInterface::class);

        foreach ($sources as $package) {
            try {
                if (preg_match('/(#[01])$/', $package, $matches)) {
                    $package  = preg_replace('/(#[01])$/', '', $package);
                    $clientId = (int)(substr($matches[1], 1));
                } else {
                    $clientId = null;
                }

                $query = $db->getQuery(true);

                if (str_starts_with($package, 'plg_')) {
                    // Plugin
                    [, $group, $plugin] = explode('_', $package, 3);
                    $isSystemCore = $group === 'system' && $plugin === 'joomlab';

                    if ($isSystemCore) {
                        // Don't uninstall the system CORE as it maybe used by other extensions
                        continue;
                    }

                    $query->select($db->quoteName('extension_id'))
                        ->from($db->quoteName('#__extensions'))
                        ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
                        ->where($db->quoteName('folder') . ' = ' . $db->quote($group))
                        ->where($db->quoteName('element') . ' = ' . $db->quote($plugin));

                    if (is_int($clientId)) {
                        $query->where($db->quoteName('client_id') . ' = ' . $db->quote($clientId));
                    }

                    if ($eid = $db->setQuery($query)->loadResult()) {
                        $installer = new Installer();
                        $installer->uninstall('plugin', $eid);
                    }
                } elseif (str_starts_with($package, 'mod_')) {
                    // Module
                    [, $module] = explode('_', $package, 2);
                    $query->select($db->quoteName('extension_id'))
                        ->from($db->quoteName('#__extensions'))
                        ->where($db->quoteName('type') . ' = ' . $db->quote('module'))
                        ->where($db->quoteName('element') . ' = ' . $db->quote($module));

                    if (is_int($clientId)) {
                        $query->where($db->quoteName('client_id') . ' = ' . $db->quote($clientId));
                    }

                    if ($eid = $db->setQuery($query)->loadResult()) {
                        $installer = new Installer();
                        $installer->uninstall('module', $eid);
                    }
                }
            } catch (Throwable $e) {
                Factory::getApplication()->enqueueMessage($e->getMessage(), 'warning');
            }
        }
    }
}