<?php

/**
 * @package     Joomlab
 * @subpackage  com_jladmin
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLAdmin\Administrator\Controller;

defined('_JEXEC') or die;

use Joomla\Archive\Zip;
use Joomla\CMS\Access\Exception\NotAllowed;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Response\JsonResponse;
use Joomla\Component\JLAdmin\Administrator\Model\BackupModel;
use Joomla\Plugin\System\Joomlab\Helper\FileSystemHelper;
use RuntimeException;
use Throwable;

class BackupController extends BaseController
{
    public function make()
    {
        try {
            if (!$this->checkToken('GET', false)) {
                throw new RuntimeException(Text::_('JINVALID_TOKEN_NOTICE'));
            }

            if (!$this->app->getIdentity()->authorise('core.admin', 'com_jladmin')) {
                throw new NotAllowed(Text::_('JERROR_ALERTNOAUTHOR'), 403);
            }

            // Set headers for SSE
            $this->app->setHeader('Content-Type', 'text/event-stream')
                ->setHeader('Content-Type', 'text/event-stream')
                ->setHeader('Cache-Control', 'no-cache')
                ->setHeader('Connection', 'keep-alive')
                ->setHeader('X-Accel-Buffering', 'no') // Prevent buffering in Nginx
                ->sendHeaders();

            if (function_exists('ini_set') && function_exists('ini_get')) {
                ini_set('memory_limit', '-1');
                ini_set('output_buffering', 'off');
                ini_set('zlib.output_compression', 'Off');

                if (function_exists('set_time_limit')) {
                    set_time_limit(0);
                }
            }

            /** @var BackupModel $backupModel */
            $backupModel = $this->getModel('Backup', 'Administrator', ['ignore_request' => true]);

            try {
                $backupModel->backup(
                    $this->input->getUint('profileId'),
                    function ($step, $data) {
                        switch ($step) {
                            case 'startingBackup':
                                $percent = 1;
                                echo 'data: ' . json_encode([
                                        'percent' => $percent,
                                        'text'    => Text::_('COM_JLADMIN_STARTING_BACKUP_MSG'),
                                        'info'    => Zip::hasNativeSupport()
                                            ? null
                                            : ['type' => 'warning', 'text' => Text::_('COM_JLADMIN_NO_ZIP_NATIVE_MSG')],
                                    ]) . PHP_EOL . PHP_EOL;
                                break;

                            case 'collectingFiles':
                                $percent = round(($data['scannedFiles'] / $data['totalFiles']) * 45, 2);
                                echo 'data: ' . json_encode([
                                        'percent' => $percent,
                                        'text'    => Text::sprintf('COM_JLADMIN_COLLECTING_FILES_COUNT_MSG', $data['scannedFiles'], $data['totalFiles']),
                                    ]) . PHP_EOL . PHP_EOL;
                                break;

                            case 'collectFilesDone':
                                echo 'data: ' . json_encode([
                                        'percent' => 45,
                                        'text'    => Text::sprintf('COM_JLADMIN_COLLECTION_FILES_DONE_MSG', $data['scannedFiles'], $data['totalFiles']),
                                        'info'    => [
                                            'type' => 'success',
                                            'text' => Text::sprintf('COM_JLADMIN_COLLECTION_FILES_DONE_MSG', $data['scannedFiles'], $data['totalFiles']),
                                        ],
                                    ]) . PHP_EOL . PHP_EOL;
                                break;

                            case 'startingBackupDatabase':
                                echo 'data: ' . json_encode(['text' => Text::_('COM_JLADMIN_STARTING_BACKUP_DATABASE_MSG')]) . PHP_EOL . PHP_EOL;
                                break;

                            case 'backupDataTableDone':
                                $percent = round(45 + (($data['scannedTables'] / $data['totalTables']) * 45), 2);
                                echo 'data: ' . json_encode([
                                        'percent' => $percent,
                                        'text'    => Text::sprintf('COM_JLADMIN_BACKING_UP_TABLE_MSG', $data['table']),

                                    ]) . PHP_EOL . PHP_EOL;
                                break;

                            case 'backupDatabaseDone':
                                echo 'data: ' . json_encode([
                                        'percent' => JLADMIN_PRO ? 65 : 95,
                                        'text'    => Text::_('COM_JLADMIN_BACKUP_DATABASE_DONE_MSG'),
                                        'info'    => [
                                            'type' => 'success',
                                            'text' => Text::_('COM_JLADMIN_BACKUP_DATABASE_DONE_MSG'),
                                        ],
                                    ]) . PHP_EOL . PHP_EOL;
                                break;

                            case 'startingCompressBackupFile':
                                echo 'data: ' . json_encode([
                                        'text' => Text::_('COM_JLADMIN_STARTING_COMPRESS_ARCHIVE_FILES_MSG'),
                                    ]) . PHP_EOL . PHP_EOL;
                                break;

                            case 'compressBackupFileDone':
                                echo 'data: ' . json_encode([
                                        'percent' => JLADMIN_PRO ? 75 : 99,
                                        'text'    => Text::_('COM_JLADMIN_COMPRESS_ARCHIVE_FILES_DONE_MSG'),
                                        'info'    => [
                                            'type' => 'success',
                                            'text' => Text::_('COM_JLADMIN_COMPRESS_ARCHIVE_FILES_DONE_MSG'),
                                        ],
                                    ]) . PHP_EOL . PHP_EOL;
                                break;

                            case 'startUploadToCloudStorage':
                                echo 'data: ' . json_encode([
                                        'text' => Text::sprintf('PLG_JLADMIN_PRO_STARTING_UPLOAD_TO_CLOUD_STORAGE_MSG', $data['name']),
                                    ]) . PHP_EOL . PHP_EOL;
                                break;


                            case 'uploadToCloudStorageSuccessfully':
                                echo 'data: ' . json_encode([
                                        'text' => Text::sprintf('PLG_JLADMIN_PRO_UPLOAD_TO_CLOUD_STORAGE_SUCCESS_MSG', $data['name']),
                                    ]) . PHP_EOL . PHP_EOL;
                                break;

                            case 'startToCheckDeleteOldFilesFromCloudStorage':
                                echo 'data: ' . json_encode([
                                        'text' => Text::sprintf('PLG_JLADMIN_PRO_STARTING_CHECK_DELETE_OLD_CLOUD_STORAGE_FILES_MSG', $data['name']),
                                    ]) . PHP_EOL . PHP_EOL;
                                break;

                            case 'checkDeleteOldFilesFromCloudStorageDone':
                                $currentFiles = implode('<br>', $data['files']['currentFiles']);
                                $deletedFiles = implode('<br>', $data['files']['deletedFiles']);

                                if (isset($data['files']['currentFilesCount'])) {
                                    $text = Text::sprintf('PLG_JLADMIN_PRO_CHECK_DELETE_OLD_CLOUD_STORAGE_FILES_DONE_MORE_MSG', $data['name'], $deletedFiles, $currentFiles,
                                        $data['files']['currentFilesCount']);
                                } else {
                                    $text = Text::sprintf('PLG_JLADMIN_PRO_CHECK_DELETE_OLD_CLOUD_STORAGE_FILES_DONE_MSG', $data['name'], $deletedFiles, $currentFiles);
                                }

                                echo 'data: ' . json_encode([
                                        'text' => Text::sprintf('PLG_JLADMIN_PRO_STARTING_CHECK_DELETE_OLD_CLOUD_STORAGE_FILES_MSG', $data['name']),
                                        'info' => [
                                            'type' => 'success',
                                            'text' => nl2br($text),
                                        ],
                                    ]) . PHP_EOL . PHP_EOL;

                                if (!empty($data['files']['warningFiles'])) {
                                    echo 'data: ' . json_encode([
                                            'info' => [
                                                'type' => 'warning',
                                                'text' => implode('<br>', $data['files']['warningFiles']),
                                            ],
                                        ]) . PHP_EOL . PHP_EOL;
                                }

                                break;

                            case 'uploadToCloudStorageError':
                                echo 'data: ' . json_encode([
                                        'percent' => JLADMIN_PRO ? 65 : 95,
                                        'info'    => [
                                            'type' => 'warning',
                                            'text' => $data['message'],
                                        ],
                                    ]) . PHP_EOL . PHP_EOL;
                                break;
                        }

                        $this->flush();
                    }
                );

                echo 'data: ' . json_encode(['percent' => 100, 'text' => Text::_('COM_JLADMIN_BACKUP_DONE_MSG')]) . PHP_EOL . PHP_EOL;
            } catch (Throwable $e) {
                echo 'data: ' . json_encode(['percent' => 100, 'text' => $e->getMessage()]) . PHP_EOL . PHP_EOL;
            }

            $this->flush();
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    private function flush()
    {
        // flush the output buffer and send echoed messages to the browser
        while (ob_get_level() > 0) {
            ob_end_flush();
        }

        flush();

        // Small delay to prevent CPU overload
        usleep(1000); // 1ms delay
    }

    public function download()
    {
        $this->checkToken('request');

        if (!$this->app->getIdentity()->authorise('core.admin', 'com_jladmin')) {
            throw new NotAllowed(Text::_('JERROR_ALERTNOAUTHOR'), 403);
        }

        $backupId = $this->input->getUint('backupId', 0);
        $backup   = $this->getModel('Backup', 'Administrator', ['ignore_request' => true])
            ->getItem($backupId);

        if (!$backup->id) {
            throw new RuntimeException('Backup not found');
        }

        FileSystemHelper::stream(JPATH_ADMINISTRATOR . '/components/com_jladmin/backups/' . $backup->backupFile);
    }
}
