<?php

/**
 * @package     Joomlab
 * @subpackage  com_jladmin
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLAdmin\Administrator\Controller;

defined('_JEXEC') or die;

use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Response\JsonResponse;
use Joomla\Component\JLAdmin\Administrator\Trait\PathTrait;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\Filesystem\Path;
use Joomla\Plugin\System\Joomlab\Helper\FileSystemHelper;
use Joomla\Plugin\System\Joomlab\Helper\LayoutHelper;
use Joomla\Plugin\System\Joomlab\Helper\ZipHelper;
use RuntimeException;
use Throwable;

class SourceController extends BaseController
{
    use PathTrait;

    public function readPath()
    {
        try {
            $path     = $this->getPath();
            $fullPath = JPATH_ROOT . '/' . $path;
            $isFile   = is_file($fullPath);
            $mime     = null;

            if ($isFile && false === ($mime = mime_content_type($fullPath))) {
                throw new RuntimeException(Text::sprintf('COM_JLADMIN_UNKNOWN_FILE_TYPE_MSG', $path));
            }

            $fileInfo = LayoutHelper::render(
                'source.file-info',
                [
                    'source' => $fullPath,
                    'type'   => $isFile ? 'file' : 'folder',
                ]
            );
            $tree     = LayoutHelper::render(
                'source.tree',
                $this->getModel('Source', 'Administrator')->getData($isFile ? dirname($fullPath) : $fullPath)
            );

            if ($isFile) {
                if (str_starts_with($mime, 'image/') || str_starts_with($mime, 'video/')) {
                    FileSystemHelper::stream($fullPath);
                }

                if ($this->isReadableFile($mime)) {
                    echo new JsonResponse([
                        'type' => 'file',
                        'data' => file_get_contents($fullPath),
                        'info' => $fileInfo,
                        'tree' => $tree,
                    ]);
                } else {
                    echo new JsonResponse(['type' => 'fileInfo', 'info' => $fileInfo]);
                }
            } else {
                echo new JsonResponse(
                    [
                        'type' => 'folder',
                        'tree' => $tree,
                        'info' => $fileInfo,
                    ]
                );
            }
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    public function renamePath()
    {
        try {
            $path     = $this->getPath(true);
            $fullPath = JPATH_ROOT . '/' . $path;
            $name     = $this->getInputName();

            if (empty($name)) {
                throw new RuntimeException(Text::_('COM_JLADMIN_EMPTY_INPUT_NAME_MSG'));
            }

            $newPath = dirname($fullPath) . '/' . $name;
            $isFile  = is_file($fullPath);

            if ($isFile && true !== File::move($fullPath, $newPath)) {
                throw new RuntimeException(Text::sprintf('COM_JLADMIN_CANT_RENAME_FILE_TO_NAME_MSG', $name));
            }

            if (!$isFile && true !== Folder::move($fullPath, $newPath)) {
                throw new RuntimeException(Text::sprintf('COM_JLADMIN_CANT_RENAME_FOLDER_TO_NAME_MSG', $name));
            }

            $resultPath = str_contains($path, '/') ? dirname($path) : '';
            echo new JsonResponse($resultPath, Text::sprintf('COM_JLADMIN_PATH_RENAMED_MSG', $name));
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    private function getInputName(string $input = 'name'): string
    {
        $name = Path::clean($this->input->getString($input, ''), '/');
        $name = preg_replace('/\/+/', '/', $name);
        $name = preg_replace('/\/+/', '-', $name);
        $name = File::makeSafe($name, []);

        return trim($name, '/');
    }

    public function deletePath()
    {
        try {
            $path     = $this->getPath(true);
            $fullPath = JPATH_ROOT . '/' . $path;

            if ((is_file($fullPath) && !File::delete($fullPath)) || (is_dir($fullPath) && !Folder::delete($fullPath))) {
                throw new RuntimeException(Text::sprintf('COM_JLADMIN_CANT_DELETE_PATH_MSG', $path));
            }

            $paths    = explode('/', Path::clean($path, '/'));
            $prevPath = '';

            if (count($paths) > 1) {
                array_pop($paths);
                $prevPath = implode('/', $paths);
            }

            echo new JsonResponse($prevPath, Text::sprintf('COM_JLADMIN_PATH_DELETED_MSG', $path));
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    public function createNewFile()
    {
        try {
            $fileName = $this->getInputName();

            if (empty($fileName)) {
                throw new RuntimeException(Text::_('COM_JLADMIN_EMPTY_INPUT_NAME_MSG'));
            }

            $fileName = explode('/', $fileName)[0];
            $path     = $this->getPath();
            $fullPath = JPATH_ROOT . '/' . $path . '/' . $fileName;
            $ext      = '';

            if (str_contains($fileName, '.')) {
                $parts = explode('.', $fileName);
                $ext   = array_pop($parts);
            }

            $buffer = match (strtolower($ext)) {
                'php' => '<?php' . PHP_EOL,
                'xml' => '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL,
                'html' => '<!DOCTYPE html>' . PHP_EOL,
                default => '# EMPTY FILE' . PHP_EOL,
            };

            if (!File::write($fullPath, $buffer)) {
                throw new RuntimeException(Text::sprintf('COM_JLADMIN_CANT_CREATE_FILE_MSG', $path . '/' . $fileName));
            }

            echo new JsonResponse($path . '/' . $fileName, Text::sprintf('COM_JLADMIN_PATH_CREATED_MSG', $path . '/' . $fileName));
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    public function createNewFolder()
    {
        try {
            $path       = $this->getPath();
            $folderName = $this->getInputName();

            if (empty($folderName)) {
                throw new RuntimeException(Text::_('COM_JLADMIN_EMPTY_INPUT_NAME_MSG'));
            }

            $folderName = explode('/', $folderName)[0];
            $baseDir    = $path ? $path . '/' : '';
            $fullPath   = JPATH_ROOT . '/' . $baseDir . $folderName;
            $basePath   = $baseDir . $folderName;

            if (!Folder::create($fullPath)) {
                throw new RuntimeException(Text::sprintf('COM_JLADMIN_CANT_CREATE_FOLDER_MSG', $basePath));
            }

            echo new JsonResponse($path, Text::sprintf('COM_JLADMIN_PATH_CREATED_MSG', $basePath));
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    public function downloadPath()
    {
        FileSystemHelper::stream(JPATH_ROOT . '/' . $this->getPath());
    }

    public function compressPath()
    {
        try {
            $path     = $this->getPath(true);
            $fullPath = JPATH_ROOT . '/' . $path;
            $archive  = $fullPath . '.zip';
            FileSystemHelper::increaseMemoryLimit();
            ZipHelper::compress($archive, $fullPath);

            echo new JsonResponse(trim(dirname($path), './'), Text::sprintf('COM_JLADMIN_PATH_COMPRESSED_TO_MSG', basename($archive)));
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    public function extractPath()
    {
        try {
            $path        = $this->getPath(true);
            $archive     = JPATH_ROOT . '/' . $path;
            $destination = $this->getInputName('destination');
            $dir         = dirname($path);

            if ($destination) {
                $resultPath  = dirname(($dir ? $dir . '/' : '') . $destination);
                $destination = dirname($archive) . '/' . $destination;
            } else {
                $resultPath  = $dir;
                $destination = dirname($archive) . '/' . preg_replace('/\.(zip|tar)$/i', '-$1', $path);
            }

            if (is_dir($destination)) {
                throw new RuntimeException(Text::sprintf('COM_JLADMIN_DESTINATION_EXISTS_MSG', $path . '/' . $destination));
            }

            FileSystemHelper::increaseMemoryLimit();
            ZipHelper::extract($archive, $destination);
            echo new JsonResponse(trim($resultPath, './'), Text::sprintf('COM_JLADMIN_PATH_EXTRACTED_TO_MSG', basename($destination)));
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    public function upload()
    {
        try {
            $path          = $this->getPath();
            $files         = $this->input->files->get('file', [], 'RAW');
            $errorsFiles   = [];
            $uploadedFiles = [];

            foreach ($files as $file) {
                $name = File::makeSafe($file['name']);

                if ($file['error'] !== UPLOAD_ERR_OK) {
                    $errorsFiles[] = ['name' => $name, 'error' => $file['error']];
                    continue;
                }

                $src            = $file['tmp_name'];
                $dest           = JPATH_ROOT . '/' . $path . '/' . $name;
                $nameWithoutExt = FileSystemHelper::stripExt($name);
                $ext            = FileSystemHelper::getExt($name);
                $cloneNum       = 1;

                while (is_file($dest)) {
                    $dest = JPATH_ROOT . '/' . $path . '/' . $nameWithoutExt . '-copy' . (++$cloneNum) . ($ext ? '.' . $ext : '');
                }

                try {
                    File::upload($src, $dest);
                    $uploadedFiles[] = $name;
                } catch (Throwable $e) {
                    $errorsFiles[] = ['name' => $name, 'error' => $e->getMessage()];
                }
            }

            echo new JsonResponse([
                'errorsFiles'   => $errorsFiles,
                'uploadedFiles' => $uploadedFiles,
            ]);
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    public function pastePath()
    {
        try {
            $fromPath = $this->getPath(true, 'fromPath');
            $toPath   = $this->getPath(false, 'toPath');
            $move     = !!$this->input->getUint('move');

            if (!is_dir(JPATH_ROOT . '/' . $toPath)) {
                throw new RuntimeException(Text::sprintf('COM_JLADMIN_PATH_IS_NOT_FOLDER_MSG', $toPath));
            }

            $baseDir         = $toPath ? $toPath . '/' : '';
            $fromBaseName    = basename($fromPath);
            $fullPath        = JPATH_ROOT . '/' . $fromPath;
            $copyNum         = 1;
            $duplicateSuffix = '-' . ($move ? 'cut' : 'copy');

            if (is_file($fullPath)) {
                $ext            = FileSystemHelper::getExt($fromBaseName);
                $ext            = $ext ? '.' . $ext : '';
                $nameWithoutExt = FileSystemHelper::stripExt($fromBaseName);
                $dest           = JPATH_ROOT . '/' . $baseDir . $fromBaseName;

                if (is_file($dest)) {
                    $dest = JPATH_ROOT . '/' . $baseDir . $nameWithoutExt . $duplicateSuffix . $ext;

                    while (is_file($dest)) {
                        $dest = JPATH_ROOT . '/' . $baseDir . $nameWithoutExt . $duplicateSuffix . (++$copyNum) . $ext;
                    }
                }

                if ($move) {
                    File::move($fullPath, $dest);
                } else {
                    File::copy($fullPath, $dest);
                }
            } else {
                if (Path::clean($fromPath) === Path::clean($toPath) || ($toPath && str_starts_with($fromPath, $toPath))) {
                    throw new RuntimeException(Text::_('COM_JLADMIN_CANT_COPY_THE_SAME_FOLDER_MSG'));
                }

                $dest = JPATH_ROOT . '/' . $baseDir . $fromBaseName;

                if (is_dir($dest)) {
                    $dest .= $duplicateSuffix;

                    while (is_dir($dest)) {
                        $dest = JPATH_ROOT . '/' . $baseDir . $fromBaseName . $duplicateSuffix . (++$copyNum);
                    }
                }

                if (!$move) {
                    Folder::copy($fullPath, $dest, null, true);
                } elseif (true !== ($result = Folder::move($fullPath, $dest))) {
                    throw new RuntimeException($result);
                }
            }

            echo new JsonResponse($toPath, Text::_('COM_JLADMIN_PATH_' . ($move ? 'MOVED' : 'COPIED') . '_MSG'));
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    public function saveFile()
    {
        try {
            $path     = $this->getPath(true);
            $filePath = JPATH_ROOT . '/' . $path;

            if (!is_file($filePath)) {
                throw new RuntimeException(Text::_('COM_JLADMIN_PATH_IS_NOT_FILE_MSG'));
            }

            if (false === ($mime = mime_content_type($filePath))) {
                throw new RuntimeException(Text::sprintf('COM_JLADMIN_UNKNOWN_FILE_TYPE_MSG', $path));
            }

            if (!$this->isReadableFile($mime)) {
                throw new RuntimeException(Text::_('COM_JLADMIN_PATH_IS_NOT_FILE_MSG'));
            }

            $data = $this->input->get('data', '', 'RAW');

            if (!is_string($data)) {
                // Maybe attacker, don't need to translate
                throw new RuntimeException('File data must be a string');
            }

            $mode = null;

            if (false !== ($filePerms = fileperms($filePath))) {
                $mode = substr(sprintf('%o', $filePerms), -4);
            }

            if (Path::isOwner($filePath) && !Path::setPermissions($filePath, '0644')) {
                throw new RuntimeException(Text::_('COM_JLADMIN_FILE_NOT_WRITABLE_MSG'));
            }

            if (!File::write($filePath, $data)) {
                throw new RuntimeException(Text::_('COM_JLADMIN_ERROR_WRITE_FAILED'));
            }

            if ($mode && Path::isOwner($filePath)) {
                Path::setPermissions($filePath, $mode);
            }

            echo new JsonResponse(null, Text::_('COM_JLADMIN_FILE_SAVED_MSG'));
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    public function setPermission()
    {
        try {
            $path       = $this->getPath(true);
            $fullPath   = JPATH_ROOT . '/' . $path;
            $permission = trim($this->input->getString('permission', ''));
            $mode       = ltrim($permission, '0');

            if (!is_numeric($mode) || !preg_match('/^[0-7]{3,4}$/', $mode)) {
                throw new RuntimeException(Text::_('COM_JLADMIN_INVALID_PERMISSION_MSG'));
            }

            if (!Path::canChmod($fullPath) || !@chmod($fullPath, octdec($permission))) {
                throw new RuntimeException(Text::_('COM_JLADMIN_CAN_NOT_CHMOD_MSG'));
            }

            echo new JsonResponse(null, Text::_('COM_JLADMIN_SET_PERMISSION_SUCCESS_MSG'));
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }

    public function getPermission()
    {
        try {
            $path     = $this->getPath(true);
            $fullPath = JPATH_ROOT . '/' . $path;
            $perms    = fileperms($fullPath);

            // Get last 4 digits in octal format
            $permissions = substr(decoct($perms), -4);

            echo new JsonResponse($permissions);
        } catch (Throwable $e) {
            echo new JsonResponse($e);
        }
    }
}
