<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLForm\Administrator\Controller;

defined('_JEXEC') or die;

use Joomla\CMS\Access\Exception\NotAllowed;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\Mail\Mail;
use Joomla\CMS\Mail\MailerFactoryInterface;
use Joomla\CMS\MVC\Controller\FormController;
use Joomla\CMS\Response\JsonResponse;
use RuntimeException;
use Throwable;

class SubmissionController extends FormController
{
    public function renderListReplies()
    {
        $submissionId = $this->app->input->getUint('submissionId', 0);

        if (!$submissionId) {
            throw new RuntimeException('Invalid submissionId ' . $submissionId);
        }

        $repliesModel = parent::getModel('SubmissionReplies', 'Administrator');
        $repliesModel->setState('filter.submissionId', $submissionId);

        if (!$this->app->isClient('administrator')) {
            $userId = $this->app->getIdentity()->id;
            $table  = $repliesModel->getTable('Submission');

            if (!$table->load($submissionId)
                || !$userId
                || $table->createdBy != $userId
            ) {
                throw new NotAllowed(Text::_('JERROR_ALERTNOAUTHOR'), 403);
            }
        }

        $offset = $this->getRepliesFilterList();
        [$ordering, $direction] = explode(' ', $offset['order']);
        $repliesModel->setState('list.ordering', $ordering);
        $repliesModel->setState('list.direction', $direction);
        $repliesModel->setState('list.start', $offset['start']);
        $repliesModel->setState('list.limit', $offset['limit']);

        echo new JsonResponse(
            LayoutHelper::render(
                'submission.replies',
                [
                    'replies'    => $repliesModel->getItems(),
                    'pagination' => $repliesModel->getPagination(),
                ],
                JPATH_SITE . '/components/com_jlform/layouts',
                ['component' => 'com_jlform', 'client' => 'auto'],
            )
        );
    }

    protected function getRepliesFilterList(): array
    {
        return [
            'start' => $this->app->getUserStateFromRequest('com_jlform.submission.replies.start', 'start', 0, 'UINT'),
            'limit' => ComponentHelper::getParams('com_jlform')->get('submissionRepliesListLimit', 5),
            'order' => $this->app->getUserStateFromRequest('com_jlform.submission.replies.order', 'order', 'a.createdDate asc', 'STRING'),
        ];
    }

    public function saveReply()
    {
        $this->checkToken();
        $submissionReplyId = $this->app->input->getUint('submissionReplyId', 0);
        $submissionId      = $this->app->input->getUint('submissionId', 0);
        $sendClientEmails   = $this->app->input->get('sendEmails', [], 'ARRAY');
        $content           = trim($this->app->input->getString('content', ''));
        $submissionTable   = $this->factory->createTable('Submission', 'Administrator');
        $replyTable        = $this->factory->createTable('SubmissionReply', 'Administrator');

        if (empty($content)) {
            throw new RuntimeException('Content is required');
        }

        if (!$submissionId || !$submissionTable->load($submissionId)) {
            throw new RuntimeException('Invalid submissionId ' . $submissionId);
        }

        $isSite = $this->app->isClient('site');
        $user   = $this->app->getIdentity();

        if ($isSite && $submissionTable->createdBy != $user->id) {
            throw new NotAllowed(Text::_('JERROR_ALERTNOAUTHOR'), 403);
        }

        if ($submissionReplyId) {
            if (!$replyTable->load($submissionReplyId)) {
                throw new RuntimeException('Invalid submissionReplyId ' . $submissionId);
            }

            if (($isSite && $replyTable->createdBy != $user->id) || (!$isSite && !$user->authorise('core.edit', 'com_jlform'))) {
                throw new NotAllowed(Text::_('JERROR_ALERTNOAUTHOR'), 403);
            }
        }

        $replyTable->bind(['submissionId' => $submissionId, 'content' => $content]);
        $replyTable->check();

        if (!$replyTable->store()) {
            throw new RuntimeException('Cannot save the submission reply, please contact the administrator.');
        }

        $metadata   = json_decode($submissionTable->metadata ?: '{}', true);
        $recipients = null;
        $subject    = null;

        if ($isSite) {
            $formTable = $this->factory->createTable('Form', 'Administrator');

            if ($formTable->load($submissionTable->formId)
                && (json_decode($formTable->params ?: '{}', true)['notifyReplyEmail'] ?? '1')
                && ($adminEmails = $metadata['emails']['admin']['recipient'] ?? [])
            ) {
                $recipients = $adminEmails;
                $subject    = Text::sprintf('COM_JLFORM_SUBMISSION_REPLY_EMAIL_ADMIN_SUBJECT', ucfirst($metadata['emails']['admin']['subject']));
            }
        } elseif (!empty($sendClientEmails)) {
            $recipients = array_filter($sendClientEmails, fn($email) => false !== filter_var($email, FILTER_VALIDATE_EMAIL));
            $subject    = Text::sprintf('COM_JLFORM_SUBMISSION_REPLY_EMAIL_CLIENT_SUBJECT', $this->app->get('sitename'));
        }

        $message = null;
        if ($recipients && $subject) {
            try {
                /** @var Mail $mailer */
                $mailer = Factory::getContainer()->get(MailerFactoryInterface::class)->createMailer();
                $mailer->addRecipient($recipients)
                    ->isHtml()
                    ->setSubject($subject)
                    ->setBody(preg_replace('/\r\n|\n/', '<br>', $content))
                    ->send();
            } catch (Throwable $e) {
                $message = $e->getMessage();
            }
        }

        echo new JsonResponse(null, $message);
    }

    public function deleteReply()
    {
        $this->checkToken();
        $user              = $this->app->getIdentity();
        $submissionReplyId = $this->app->input->getUint('submissionReplyId', 0);
        $replyTable        = $this->factory->createTable('SubmissionReply', 'Administrator');
        $isSite            = $this->app->isClient('site');

        if ($submissionReplyId && !$replyTable->load($submissionReplyId)) {
            throw new RuntimeException('Invalid submissionReplyId ' . $submissionReplyId);
        }

        if (($isSite && $replyTable->createdBy != $user->id) || (!$isSite && !$user->authorise('core.delete', 'com_jlform'))) {
            throw new NotAllowed(Text::_('JERROR_ALERTNOAUTHOR'), 403);
        }

        if (!$replyTable->delete($submissionReplyId)) {
            throw new RuntimeException('Cannot delete the submission reply, please contact the administrator.');
        }

        echo new JsonResponse('OK');
    }
}
