<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLForm\Site\Helper;

defined('_JEXEC') or die;

use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\Registry\Registry;

abstract class FieldValidator
{
    protected array $errors = [];

    protected string $name;

    protected string $type;

    protected string $label;

    protected bool $required;

    protected string $message;

    protected Registry $options;
    protected array $postData;

    protected string $language;

    protected mixed $value = null;

    protected CMSApplicationInterface $app;
    /**
     * @var string | null
     * Accepted values: string | number | boolean | array | null
     * @since 1.0.0
     */
    protected ?string $dataType = null;
    private array $labels;

    public function __construct(array $configData, string $language)
    {
        $this->language = $language;
        $this->name     = $configData['general']['name'];
        $this->type     = $configData['general']['type'];
        $this->labels   = $configData['general']['label'] ?? [];
        $this->label    = FormHelper::_($this->labels);
        $this->message  = FormHelper::_($configData['general']['message'] ?? []);
        $this->required = (bool)($configData['general']['required'] ?? false);
        $this->options  = new Registry($configData['options'] ?? '{}');
        $this->app      = Factory::getApplication();
        $this->postData = $this->app->getInput()->post->get('JLForm', [], 'ARRAY');
    }

    public function validate(&$value = null): bool
    {
        $value  = $value ?? $this->postData[$this->name] ?? null;
        $strLen = 0;

        if (is_string($value)) {
            $value  = trim($value);
            $strLen = mb_strlen($value);
        }

        $isNothingValue = !isset($value) || $value === '';
        $isSwitcher     = $this->type === 'Switcher';

        if ((!$this->required && $isNothingValue) || $isSwitcher) {
            if ($isSwitcher) {
                $this->value = $value = (bool)$value;
            }

            return true;
        }

        if ($isNothingValue) {
            if ($this->required) {
                $this->addError(Text::_('COM_JLFORM_ERR_FIELD_REQUIRED_MSG'));

                return false;
            }

            return true;
        }

        if ($this->dataType === 'string') {
            if (!is_string($value)) {
                $this->addError(Text::_('COM_JLFORM_ERR_FIELD_INVALID_STRING_MSG'));

                return false;
            }

            $minLength = $this->options->get('minLength');
            $maxLength = $this->options->get('maxLength');
            $invalid   = false;

            if (is_numeric($minLength) && (int)$minLength > $strLen) {
                $this->addError(Text::sprintf('COM_JLFORM_ERR_FIELD_INVALID_MIN_LENGTH_MSG', $minLength));
                $invalid = true;
            }

            if (is_numeric($maxLength) && (int)$maxLength < $strLen) {
                $this->addError(Text::sprintf('COM_JLFORM_ERR_FIELD_INVALID_MAX_LENGTH_MSG', $maxLength));
                $invalid = true;
            }

            if ($invalid) {
                return false;
            }
        }

        if ($this->dataType === 'number') {
            if (!is_numeric($value)) {
                $this->addError(Text::sprintf('COM_JLFORM_ERR_FIELD_INVALID_NUMBER_MSG'));

                return false;
            }

            $value   = (float)$value;
            $minimum = $this->options->get('minimum');
            $maximum = $this->options->get('maximum');
            $invalid = false;

            if (is_numeric($minimum) && (float)$minimum > $value) {
                $this->addError(Text::sprintf('COM_JLFORM_ERR_FIELD_INVALID_MINIMUM_MSG', $minimum));
                $invalid = true;
            }

            if (is_numeric($maximum) && (float)$maximum < $value) {
                $this->addError(Text::sprintf('COM_JLFORM_ERR_FIELD_INVALID_MAXIMUM_MSG', $maximum));
                $invalid = true;
            }

            if ($invalid) {
                return false;
            }
        }

        if ($this->dataType === 'boolean') {
            if (!(is_bool($value) || in_array($value, ['true', 'false', '1', '0']))) {
                $this->addError(Text::_('COM_JLFORM_ERR_FIELD_INVALID_NUMBER_MSG'));

                return false;
            }

            if (!is_bool($value)) {
                $value = in_array($value, ['true', '1']);
            }
        }

        if (($this->dataType === 'array' || $this->options->get('multiple')) && !is_array($value)) {
            $this->addError(Text::_('COM_JLFORM_ERR_FIELD_INVALID_ARRAY_MSG'));

            return false;
        }

        $regex = $this->options->get('regex');

        if (is_string($regex) && !empty($regex)) {
            $modifiers = @preg_match('/\pL/u', 'a') ? 'u' : '';

            if (!preg_match(chr(1) . $regex . chr(1) . $modifiers, $value)) {
                $this->addError(Text::sprintf('COM_JLFORM_ERR_FIELD_INVALID_REGEX_MSG', $regex));

                return false;
            }
        }

        $this->doValidate($value);

        if ($isValid = empty($this->errors)) {
            $this->value = $value;
        }

        return $isValid;
    }

    public function addError(string $error): FieldValidator
    {
        $this->errors[] = $error;

        return $this;
    }

    abstract protected function doValidate(&$value);

    public function getErrors(): array
    {
        return $this->message ? [$this->message] : $this->errors;
    }

    public function setErrors(array $errors): FieldValidator
    {
        $this->errors = $errors;

        return $this;
    }

    public function getValue()
    {
        return $this->value;
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function getLabel(): string
    {
        return $this->label;
    }

    public function getLabels(): array
    {
        return $this->labels;
    }
}