<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLForm\Administrator\Model;

defined('_JEXEC') or die;

use DOMDocument;
use DOMXPath;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Filesystem\File;
use Joomla\Plugin\System\Joomlab\MVC\Model\AdminModel;
use PhpParser\Error;
use PhpParser\ParserFactory;

class FormModel extends AdminModel
{
    public function save($data)
    {
        $customPhp = null;

        if (!empty($data['params']['customPhp'])) {
            $customPhp = trim($data['params']['customPhp']);

            if (preg_replace('/\s+|\n+/', ' ', $customPhp) === preg_replace(
                    '/\s+|\n+/',
                    ' ',
                    trim($this->getDefaultCustomPhp())
                )) {
                unset($customPhp);
            } else {
                // Validate custom PHP
                try {
                    require_once JPATH_ADMINISTRATOR . '/components/com_jlform/lib/vendor/autoload.php';
                    (new ParserFactory())->createForNewestSupportedVersion()->parse($customPhp);
                } catch (Error $error) {
                    $this->setError('Custom PHP Error: ' . $error->getMessage());

                    return false;
                }
            }

            unset($data['params']['customPhp']);
        }


        if ($result = parent::save($data)) {
            $formId      = $this->getState('form.id');
            $customPath  = JPATH_ADMINISTRATOR . '/components/com_jlform/assets/customs';
            $phpFilePath = $customPath . '/form-' . $formId . '.php';

            if (isset($customPhp)) {
                File::write($phpFilePath, $customPhp);
            } elseif (is_file($phpFilePath)) {
                File::delete($phpFilePath);
            }
        }

        return $result;
    }

    protected function getDefaultCustomPhp(): string
    {
        return "<?php\ndefined('_JEXEC') or die;
        
use Joomla\\Component\\JLForm\\Site\\Event\\BeforeShowEvent;
use Joomla\\Component\\JLForm\\Site\\Event\\BeforeSubmitEvent;
use Joomla\\Component\\JLForm\\Site\\Event\\AfterSubmitEvent;

function onJLFormBeforeShow(BeforeShowEvent \$event) {}

function onJLFormBeforeSubmit(BeforeSubmitEvent \$event) {}
            
function onJLFormAfterSubmit(AfterSubmitEvent \$event) {}";
    }

    protected function preprocessForm(Form $form, $data, $group = 'jlform')
    {
        $language = Factory::getApplication()->getLanguage();
        foreach (PluginHelper::getPlugin($group) as $plugin) {
            $manifestPath = JPATH_PLUGINS . '/' . $group . '/' . $plugin->name . '/forms/' . $plugin->name . '.xml';

            if (is_file($manifestPath)) {
                $dom = new DOMDocument();

                if (!$dom->load($manifestPath)) {
                    continue;
                }

                $xpath = new DOMXPath($dom);
                $fields = $xpath->query('field');
                $count  = $fields->count();

                if (!$count) {
                    continue;
                }

                $elements = [];
                $language->load('plg_' . $group . '_' . $plugin->name, JPATH_PLUGINS . '/' . $group . '/' . $plugin->name);
                $ufName  = ucfirst($plugin->name);
                $useName = 'use' . $ufName;
                $useLbl  = strtoupper('PLG_' . $group . '_' . $plugin->name . '_USE_LABEL');

                for ($i = 0; $i < $count; $i++) {
                    $field = $fields->item($i);
                    $name  = $field->getAttribute('name');
                    $field->setAttribute('name', $plugin->name . ucfirst($name));
                    $showOn = $field->getAttribute('showon') ?: '';
                    $showOn = $showOn ? 'plugins.' . $plugin->name . ucfirst($showOn) . '[AND]' . $useName . ':1' : $useName . ':1';
                    $field->setAttribute('showon', $showOn);
                    $elements[] = $dom->saveXML($field);
                }

                $elements = implode(PHP_EOL, $elements);
                $xml      = <<<XML
<form>
    <fields name="plugins">
        <fieldset name="plugins">
            <field
                name="$useName"
                type="radio"
                label="$useLbl"
                layout="joomla.form.field.radio.switcher"
                default="0"
            >
                <option value="0">JNO</option>
                <option value="1">JYES</option>
            </field>
            $elements
            <field
                type="spacer"
                hr="true"
            />
        </fieldset>
    </fields>
</form>
XML;
                $form->load(trim($xml));
                $this->setState('form.hasPlugins', true);
            }
        }
    }

    protected function preprocessData($context, &$data, $group = 'jlform')
    {
        if (!empty($data->plugins) && is_string($data->plugins)) {
            $data->plugins = json_decode($data->plugins, true);
        }

        $isGetMethod = Factory::getApplication()->getInput()->getMethod() === 'GET';

        if (!$isGetMethod || !empty($data->params['customPhp'])) {
            return;
        }

        $formId        = $data->id ?? '0';
        $customPhpFile = JPATH_ADMINISTRATOR . '/components/com_jlform/assets/customs/form-' . $formId . '.php';

        if (!is_array($data->params)) {
            $data->params = [];
        }

        $data->params['customPhp'] = $formId && is_file($customPhpFile)
            ? file_get_contents(
                $customPhpFile
            ) : $this->getDefaultCustomPhp();
    }

    public function delete(&$pks)
    {
       if ($result = parent::delete($pks)) {
           foreach ($pks as $pk) {
               $customPhpFile = JPATH_ADMINISTRATOR . '/components/com_jlform/assets/customs/form-' . $pk . '.php';

               if (is_file($customPhpFile)) {
                   File::delete($customPhpFile);
               }
           }
       }

       return $result;
    }
}
