<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLForm\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;
use Joomla\Database\QueryInterface;

class FormsModel extends ListModel
{
    public function __construct($config = [])
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = [
                'id',
                'a.id',
                'title',
                'a.title',
                'state',
                'a.state',
                'createdBy',
                'a.createdBy',
                'createdDate',
                'a.createdDate',
                'modifiedBy',
                'a.modifiedBy',
                'modifiedDate',
                'a.modifiedDate',
                'totalSubmissions',
            ];
        }

        parent::__construct($config);
    }

    protected function populateState($ordering = 'a.id', $direction = 'asc')
    {
        $search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
        $this->setState('filter.search', $search);

        $published = $this->getUserStateFromRequest($this->context . '.filter.published', 'filter_published', '');
        $this->setState('filter.published', $published);

        $submittedBy = $this->getUserStateFromRequest($this->context . '.filter.submittedBy', 'filter_submittedBy', '');
        $this->setState('filter.submittedBy', $submittedBy);

        // List state information.
        parent::populateState($ordering, $direction);
    }

    protected function getStoreId($id = ''): string
    {
        // Compile the store id.
        $id .= ':' . $this->getState('filter.search');
        $id .= ':' . $this->getState('filter.published');
        $id .= ':' . $this->getState('filter.submittedBy');

        return parent::getStoreId($id);
    }


    protected function getListQuery(): QueryInterface
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);
        $query->select(
            $this->getState(
                'list.select',
                'a.id, a.title, a.state, a.description, a.params, a.createdBy, a.createdDate, a.modifiedBy, a.modifiedDate, a.checkedOut, a.checkedOutTime'
            )
        )
            ->select('u1.name AS authorName, u2.name AS editorName')
            ->from($db->quoteName('#__joomlab_forms', 'a'))
            ->join('LEFT', $db->quoteName('#__users', 'u1'), 'u1.id = a.createdBy')
            ->join('LEFT', $db->quoteName('#__users', 'u2'), 'u2.id = a.checkedOut');

        $countQuery = $db->getQuery(true);
        $countQuery->select('COUNT(a2.id)')
            ->from($db->quoteName('#__joomlab_form_submissions', 'a2'))
            ->where('a2.formId = a.id');

        if (is_numeric($submittedBy = $this->getState('filter.submittedBy'))) {
            $countQuery->where('(a2.createdBy > 0 AND a2.createdBy = ' . (int)$submittedBy . ')');
            $query->having('totalSubmissions > 0');
        }

        $query->select('(' . $countQuery . ') as totalSubmissions');

        // Filter by published state
        $published = $this->getState('filter.published', '');

        if (is_numeric($published)) {
            $query->where($db->quoteName('a.state') . ' = :published')
                ->bind(':published', $published, ParameterType::INTEGER);
        } elseif ($published === '') {
            $query->whereIn($db->quoteName('a.state'), [0, 1]);
        }

        // Filter by search in title.
        $search = $this->getState('filter.search');

        if (!empty($search)) {
            if (stripos($search, 'id:') === 0) {
                $search = (int)substr($search, 3);
                $query->where($db->quoteName('a.id') . ' = :search')
                    ->bind(':search', $search, ParameterType::INTEGER);
            } elseif (stripos($search, 'author:') === 0) {
                $search = '%' . substr($search, 7) . '%';
                $query->where($db->quoteName('u1.name') . ' LIKE :search')
                    ->bind(':search', $search);
            } else {
                $search = '%' . trim($search) . '%';
                $query->where($db->quoteName('a.title') . ' LIKE :search')
                    ->bind(':search', $search);
            }
        }

        // Add the list ordering clause.
        $orderCol  = $this->state->get('list.ordering', 'a.id');
        $orderDirn = $this->state->get('list.direction', 'ASC');
        $query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

        return $query;
    }
}
