<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLForm\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

class SubmissionsModel extends ListModel
{
    public function __construct($config = [])
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = [
                'id',
                'a.id',
                'formId',
                'a.formId',
                'language',
                'a.language',
                'createdBy',
                'a.createdBy',
                'createdDate',
                'a.createdDate',
                'formTitle',
                'authorName',
                'totalReplies',
            ];
        }

        parent::__construct($config);
    }

    public function getFilterForm($data = [], $loadData = true)
    {
        Form::addFormPath(JPATH_ADMINISTRATOR . '/components/com_jlform/forms');
        $filterForm = parent::getFilterForm($data, $loadData);

        if ($filterForm && Factory::getApplication()->isClient('site')) {
            $filterForm->removeField('fullordering', 'list');
        }

        return $filterForm;
    }

    public function getActiveFilters()
    {
        $activeFilters = parent::getActiveFilters();

        if (Factory::getApplication()->isClient('site')) {
            unset($activeFilters['createdBy']);
        }

        return $activeFilters;
    }

    protected function populateState($ordering = 'a.createdDate', $direction = 'desc')
    {
        $search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
        $this->setState('filter.search', $search);

        $formId = $this->getUserStateFromRequest($this->context . '.filter.formId', 'filter_formId', '');
        $this->setState('filter.formId', $formId);

        $createdBy = $this->getUserStateFromRequest($this->context . '.filter.createdBy', 'filter_createdBy', '');
        $this->setState('filter.createdBy', $createdBy);

        $hasUser = $this->getUserStateFromRequest($this->context . '.filter.hasUser', 'filter_hasUser', '');
        $this->setState('filter.hasUser', $hasUser);

        $paymentStatus = $this->getUserStateFromRequest($this->context . '.filter.paymentStatus', 'filter_paymentStatus', '');
        $this->setState('filter.paymentStatus', $paymentStatus);

        // List state information.
        parent::populateState($ordering, $direction);
    }

    protected function getStoreId($id = '')
    {
        // Compile the store id.
        $id .= ':' . $this->getState('filter.search');
        $id .= ':' . $this->getState('filter.formId');
        $id .= ':' . $this->getState('filter.createdBy');
        $id .= ':' . $this->getState('filter.hasUser');
        $id .= ':' . $this->getState('filter.paymentStatus');

        return parent::getStoreId($id);
    }

    protected function getListQuery()
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);
        $query->select(
            $this->getState(
                'list.select',
                'a.id, a.formId, a.submissionData, a.metadata, a.createdBy, a.createdDate, a.language, a.checkedOut, a.checkedOutTime, a.language, a2.title AS formTitle'
            )
        )
            ->select('u1.name AS authorName, u2.name AS editorName')
            ->from($db->quoteName('#__joomlab_form_submissions', 'a'))
            ->join('INNER', $db->quoteName('#__joomlab_forms', 'a2'), 'a2.id = a.formId')
            ->join('LEFT', $db->quoteName('#__users', 'u1'), 'u1.id = a.createdBy')
            ->join('LEFT', $db->quoteName('#__users', 'u2'), 'u2.id = a.checkedOut');

        $countQuery = $db->getQuery(true);
        $countQuery->select('COUNT(a3.id)')
            ->from($db->quoteName('#__joomlab_form_submission_replies', 'a3'))
            ->where('a3.submissionId = a.id');
        $query->select('(' . $countQuery . ') as totalReplies');

        // Filter by formId state
        $formId = (int)$this->getState('filter.formId');

        if ($formId > 0) {
            $query->where($db->quoteName('a.formId') . ' = ' . $formId);
        }

        // Filter by author state
        $createdBy = $this->getState('filter.createdBy', '');

        if (is_numeric($createdBy)) {
            $query->where($db->quoteName('a.createdBy') . ' = ' . (int)$createdBy);
        }

        if ($this->getState('filter.hasUser', '')) {
            $query->where($db->quoteName('a.createdBy') . ' > 0');
        }

        // Filter by payment status
        $validPaymentStatuses = ['Pending', 'Cancelled', 'Deposited', 'Paid', 'Refunded'];
        $paymentStatus        = $this->getState('filter.paymentStatus', '');

        if ($paymentStatus && in_array($paymentStatus, $validPaymentStatuses)) {
            $query->join('INNER', $db->quoteName('#__joomlab_form_submission_payments', 'a4'), 'a4.submissionId = a.id')
                ->where('a4.status = :paymentStatus')
                ->bind(':paymentStatus', $paymentStatus);
        }

        // Filter by search in title.
        $search = $this->getState('filter.search');

        if (!empty($search)) {
            if (stripos($search, 'id:') === 0) {
                $search = (int)substr($search, 3);
                $query->where($db->quoteName('a.id') . ' = :search')
                    ->bind(':search', $search, ParameterType::INTEGER);
            } elseif (stripos($search, 'author:') === 0) {
                $search = '%' . substr($search, 7) . '%';
                $query->where($db->quoteName('u1.name') . ' LIKE :search')
                    ->bind(':search', $search);
            } else {
                $search = '%' . trim($search) . '%';
                $query->where(
                    '(' . $db->quoteName(
                        'a2.title'
                    ) . ' LIKE :search OR u1.name LIKE :search OR u1.username LIKE :search)'
                )
                    ->bind(':search', $search);
            }
        }

        // Add the list ordering clause.
        $orderCol  = $this->state->get('list.ordering', 'a.id');
        $orderDirn = $this->state->get('list.direction', 'ASC');
        $query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

        return $query;
    }
}
