<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLForm\Site\Helper\FieldValidator;

defined('_JEXEC') or die;

use DateInterval;
use DateTime;
use DateTimeZone;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\Component\JLForm\Site\Helper\FieldValidator;
use Throwable;

class CalendarField extends FieldValidator
{
    public function validate(&$value = null): bool
    {
        $value = $value ?? $this->postData[$this->name] ?? null;
        $mode  = $this->options->get('mode', 'single');

        if ('single' !== $mode) {
            $this->options->set('multiple', true);

            if ('range' === $mode
                && is_string($value)
                && !empty($value)
            ) {
                $rangeSeparator = $this->app->getInput()->post->getString('rangeSeparator', ' to ');
                $dateRange      = explode($rangeSeparator, $value, 2);
                $value          = [trim($dateRange[0]), trim($dateRange[1] ?? $dateRange[0])];
            }
        }

        return parent::validate($value);
    }

    protected function doValidate(&$value)
    {
        $dates  = is_array($value) ? $value : [$value];
        $userTz = $this->app->getIdentity()->getTimezone();
        $tz     = $this->app->getInput()->post->getString('userTimeZone', $userTz);
        $tzMaps = ['Asia/Saigon' => 'Asia/Ho_Chi_Minh'];

        if (isset($tzMaps[$tz])) {
            $tz = $tzMaps[$tz];
        }

        if (!in_array($tz, DateTimeZone::listIdentifiers())) {
            $tz = $userTz;
        }

        $parseMinMaxDate = function (string $type, DateTime $dateValue) use ($tz) {
            $date       = $this->options->get($type . 'DateFromNow');
            $dateFormat = $this->options->get('dateFormat');

            if (is_numeric($date)) {
                $date = (int)$date;
                $now  = Factory::getDate('now', $tz);

                if ($date !== 0) {
                    $now->add(new DateInterval('P' . $date . 'D'));
                }

                if ($type === 'min' && $dateValue < $now) {
                    $this->addError(
                        Text::sprintf(
                            'COM_JLFORM_ERR_FIELD_CALENDAR_TYPE_MIN_DATE_MSG',
                            $now->format($dateFormat, true)
                        )
                    );
                }

                if ($type === 'max' && $dateValue > $now) {
                    $this->addError(
                        Text::sprintf(
                            'COM_JLFORM_ERR_FIELD_CALENDAR_TYPE_MAX_DATE_MSG',
                            $now->format($dateFormat, true)
                        )
                    );
                }
            } else {
                $year       = $this->options->get($type . 'Year');
                $parsedDate = [];

                if (is_numeric($year)) {
                    $month        = $this->options->get($type . 'Month');
                    $parsedDate[] = $year;

                    if (is_numeric($month)) {
                        $date         = $this->options->get($type . 'Date');
                        $parsedDate[] = str_pad($month, 2, '0', STR_PAD_LEFT);

                        if (is_numeric($date)) {
                            $parsedDate[] = str_pad($date, 2, STR_PAD_LEFT);
                        }
                    }
                }

                if (count($parsedDate)) {
                    if (!isset($parsedDate[1])) {
                        $parsedDate[1] = '01';
                    }

                    if (!isset($parsedDate[2])) {
                        $parsedDate[2] = '01';
                    }

                    $parsedDate = Factory::getDate(implode('-', $parsedDate), $tz);

                    if ($type === 'min' && $dateValue < $parsedDate) {
                        $this->addError(
                            Text::sprintf(
                                'COM_JLFORM_ERR_FIELD_CALENDAR_TYPE_MIN_DATE_MSG',
                                $parsedDate->format($dateFormat, true)
                            )
                        );
                    }

                    if ($type === 'max' && $dateValue > $parsedDate) {
                        $this->addError(
                            Text::sprintf(
                                'COM_JLFORM_ERR_FIELD_CALENDAR_TYPE_MAX_DATE_MSG',
                                $parsedDate->format($dateFormat, true)
                            )
                        );
                    }
                }
            }

            return $date;
        };

        foreach ($dates as $i => $date) {
            try {
                $date = Factory::getDate($date, $tz);
                $parseMinMaxDate('min', $date);
                $parseMinMaxDate('Max', $date);

                // Update date for SQL value
                $dates[$i] = $date->toSql();
            } catch (Throwable $e) {
                $this->addError(Text::sprintf('COM_JLFORM_ERR_FIELD_CALENDAR_TYPE_INVALID_DATE_MSG', $date));
            }
        }

        if (empty($this->errors)) {
            // Update date value
            $value = is_array($value) ? $dates : $dates[0];
        }
    }
}
