<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLForm\Site\Traits;

defined('_JEXEC') or die;

use DateTimeZone;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Mail\Mail;
use Joomla\CMS\Mail\MailerFactoryInterface;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Response\JsonResponse;
use Joomla\CMS\Session\Session;
use Joomla\Component\JLForm\Administrator\Table\SubmissionTable;
use Joomla\Component\JLForm\Site\Event\AfterSubmitEvent;
use Joomla\Component\JLForm\Site\Event\BeforeSubmitEvent;
use Joomla\Component\JLForm\Site\Event\FormEvent;
use Joomla\Component\JLForm\Site\Event\PaymentAmountEvent;
use Joomla\Component\JLForm\Site\Helper\FieldValidator;
use Joomla\Component\JLForm\Site\Helper\FormHelper;
use Joomla\Filesystem\File;
use Joomla\Plugin\System\Joomlab\Helper\AsyncHelper;
use Joomla\Utilities\ArrayHelper;
use Joomla\Utilities\IpHelper;
use RuntimeException;
use Throwable;

trait HandleFormTrait
{
    protected array $formValidData = [];

    protected array $formDataErrors = [];

    public function processFormData($formId = null)
    {
        if (!Session::checkToken()) {
            throw new RuntimeException(Text::_('JINVALID_TOKEN'));
        }

        try {
            $app        = Factory::getApplication();
            $mvcFactory = $app->bootComponent('com_jlform')->getMVCFactory();
            $post       = $app->getInput()->post;

            if (!$formId) {
                $formId = $post->get('formId', 0, 'UINT');
            }

            $form = FormHelper::getForm($formId);

            if ($form->params->get('requireUserLogin') && $app->getIdentity()->guest) {
                throw new RuntimeException(Text::_('COM_JLFORM_FORM_REQUIRE_USER_LOGIN_TEXT'));
            }

            $step     = $post->get('step');
            $language = $app->getLanguage()->getTag();

            if (is_numeric($step)) {
                $formData = $form->data[$step] ?? null;

                if (empty($formData)) {
                    throw new RuntimeException(Text::_('COM_JLFORM_ERR_FIELD_INVALID_FORM_DATA'));
                }

                $formData = [$formData];
                $isSubmit = false;
            } else {
                $formData = $form->data;
                $isSubmit = true;
            }

            $this->formValidData  = [];
            $this->formDataErrors = [];

            foreach ($formData as $i => $step) {
                if (empty($step['rows'])) {
                    continue;
                }

                if (!isset($dataValue[$i])) {
                    $dataValue[$i] = [];
                }

                foreach ($step['rows'] as $row) {
                    if (empty($row)) {
                        continue;
                    }

                    foreach ($row as $column) {
                        foreach ($column['data'] as $field) {
                            if (empty($field['general']['type']) || empty($field['general']['name'])) {
                                continue;
                            }

                            $classValidator = FieldValidator::class . '\\' . $field['general']['type'] . 'Field';

                            if (!class_exists($classValidator) || !is_subclass_of($classValidator, FieldValidator::class)) {
                                $app->enqueueMessage(
                                    'Class ' . $classValidator . ' not found, please contact the administrator.',
                                    CMSApplicationInterface::MSG_WARNING
                                );
                                continue;
                            }

                            $fieldValidator = new $classValidator($field, $language);

                            if ($fieldValidator->validate()) {
                                $this->formValidData[$fieldValidator->getName()] = [
                                    'label'  => $fieldValidator->getLabel(),
                                    'labels' => $fieldValidator->getLabels(),
                                    'value'  => $fieldValidator->getValue(),
                                    'data'   => $field, // History field data
                                ];
                            } else {
                                $this->formDataErrors[$fieldValidator->getName()] = $fieldValidator->getErrors();
                            }
                        }
                    }
                }
            }

            $response = [
                'errors'      => $this->formDataErrors,
                'sid'         => 0,
                'media'       => [],
                'eventBuffer' => [],
                'redirect'    => null,
                'payment'     => null,
            ];

            if ($isSubmit) {
                // Validate payment plugin
                $paymentPlugins = [];
                foreach ($form->plugins->toArray() as $k => $v) {
                    if (str_ends_with($k, 'PaymentPlugin')) {
                        $plgName = str_replace('PaymentPlugin', '', $k);

                        if (PluginHelper::isEnabled('jlform', $plgName) && $form->plugins->get(
                                'use' . ucfirst($plgName)
                            )) {
                            $paymentPlugins[] = $plgName;
                        }
                    }
                }
            }
            $paymentName = $post->get('JLForm', [], 'ARRAY')['__payment'] ?? null;

            if (empty($response['errors']) && $isSubmit) {

                if ($paymentPlugins && !in_array($paymentName, $paymentPlugins, true)) {
                    $response['errors']['__payment'] = [
                        Text::_('COM_JLFORM_ERROR_INVALID_PAYMENT_MSG'),
                    ];
                }

                // Check recaptcha
                $config             = ComponentHelper::getParams('com_jlform');
                $recaptchaVersion   = (int)$config->get('useRecaptcha');
                $recaptchaSecretKey = trim($config->get('recaptchaSecretKey' . $recaptchaVersion, ''));
                $useRecaptcha       = $form->params->get('useRecaptcha', '0')
                    && in_array($recaptchaVersion, [2, 3])
                    && $recaptchaSecretKey;

                if ($useRecaptcha) {
                    $isValidCaptcha = false;

                    if ($recaptchaToken = $post->getString('g-recaptcha-response')) {
                        $recaptchaResponse = HttpFactory::getHttp()
                            ->post('https://www.google.com/recaptcha/api/siteverify', [
                                'secret'   => $recaptchaSecretKey,
                                'response' => $recaptchaToken,
                                'remoteip' => IpHelper::getIp(),
                            ]);
                        $responseJson      = json_decode($recaptchaResponse->body, true);
                        $isValidCaptcha    = !empty($responseJson['success']);

                        if ($isValidCaptcha && $recaptchaVersion === 3) {
                            $score          = (float)($responseJson['score'] ?? 0.0);
                            $isValidCaptcha = $score >= (float)$config->get('recaptchaScore', '0.5');
                        }
                    }

                    if (!$isValidCaptcha) {
                        $response['errors']['__system'] = [
                            Text::_('COM_JLFORM_INVALID_RECAPTCHA_MSG'),
                        ];
                    }
                }

                $terms = (int)($post->get('JLForm', [], 'ARRAY')['__termsAndCondition'] ?? 0);

                if (!empty($form->termsAndConditions) && !$terms) {
                    $response['errors']['__system'] = [
                        Text::_('COM_JLFORM_INVALID_TERMS_N_CONDITIONS_MSG'),
                    ];
                }
            }

            // Check again
            if (empty($response['errors']) && $isSubmit) {
                $customPhpFile = JPATH_ADMINISTRATOR . '/components/com_jlform/assets/customs/form-' . $formId . '.php';

                if (is_file($customPhpFile)) {
                    include_once $customPhpFile;
                }

                $beforeSubmitEvent = new BeforeSubmitEvent(['subject' => $form, 'validData' => $this->formValidData]);
                $app->getDispatcher()->dispatch(FormEvent::BEFORE_SUBMIT, $beforeSubmitEvent);

                if (function_exists(FormEvent::BEFORE_SUBMIT)) {
                    call_user_func_array(FormEvent::BEFORE_SUBMIT, [$beforeSubmitEvent]);
                }

                $response['errors']   = $beforeSubmitEvent->getErrors();
                $response['redirect'] = $beforeSubmitEvent->getRedirect();

                // If a redirect URL is present, we stop the submission
                if (empty($response['errors']) && empty($response['redirect'])) {
                    // Handle upload files
                    $uploadedFiles     = [];
                    $ignoreAttachments = [];
                    $formValidData     = $beforeSubmitEvent->getValidData();

                    foreach ($formValidData as $name => $validData) {
                        if ($validData['data']['general']['type'] !== 'Upload') {
                            continue;
                        }

                        $formValidData[$name]['value'] = [];

                        if (!empty($validData['value'])) {
                            $addAttachment = $validData['data']['options']['emailAttachment'] ?? false;
                            foreach ($validData['value'] as $file) {
                                $fileName                                     = File::makeSafe($file['name']);
                                $fileHashName                                 = md5(uniqid($fileName)) . '_' . $fileName;
                                $formValidData[$name]['value'][$fileHashName] = $fileName;
                                $uploadedFiles[$fileName]                     = [$fileHashName, $file['tmp_name']];

                                if (!$addAttachment) {
                                    $ignoreAttachments[] = $fileName;
                                }
                            }
                        }
                    }

                    // Hidden fields (based on show-on feature)
                    $hiddenFields  = (array)($post->get('JLForm', [], 'ARRAY')['__hiddenFields'] ?? []);
                    $allNameFields = array_keys($formValidData);

                    foreach ($hiddenFields as $i => $hiddenField) {
                        if (!in_array($hiddenField, $allNameFields)) {
                            unset($hiddenFields[$i]);
                        }
                    }

                    // Handle submit data
                    /** @var SubmissionTable $submissionTable */
                    $submissionTable = $mvcFactory->createTable('Submission', 'Administrator');
                    $user            = $app->getIdentity();
                    $userTz          = $user->getTimezone();
                    $tz              = $post->get('userTimeZone', $userTz, 'string');
                    $tzMaps          = ['Asia/Saigon' => 'Asia/Ho_Chi_Minh'];

                    if (isset($tzMaps[$tz])) {
                        $tz = $tzMaps[$tz];
                    }

                    if (!in_array($tz, DateTimeZone::listIdentifiers())) {
                        $tz = $userTz;
                    }

                    $submissionData = [
                        'formId'          => $formId,
                        'formHistoryData' => $form->data,
                        'submissionData'  => [
                            'data'          => $formValidData,
                            'allNameFields' => $allNameFields,
                            'hiddenFields'  => array_values($hiddenFields),
                        ],
                        'createdBy'       => $user->id ?: 0,
                        'createdDate'     => Factory::getDate()->toSql(),
                        'language'        => $language,
                        'userTimeZone'    => $tz,
                        'userIP'          => IpHelper::getIp(),
                        'finished'        => 0,
                        'metadata'        => [
                            'emails' => [
                                'client' => ['recipient' => [], 'subject' => null, 'body' => null],
                                'admin'  => ['recipient' => [], 'subject' => null, 'body' => null],
                            ],
                        ],
                    ];

                    // Handle mailer
                    if ($form->params->get('enableClientEmail') || $form->params->get('enableAdminEmail')) {
                        $formData       = [];
                        $clientBodyText = (array)$form->params->get('clientEmailBody', []);
                        $clientBody     = Text::_(
                            trim($clientBodyText[$language] ?? $clientBodyText['en-GB'] ?? '') ?: 'COM_JLFORM_CLIENT_DEFAULT_EMAIL_BODY'
                        );

                        $adminBodyText = (array)$form->params->get('adminEmailBody', []);
                        $adminBody     = Text::_(
                            trim($adminBodyText[$language] ?? $adminBodyText['en-GB'] ?? '') ?: 'COM_JLFORM_ADMIN_DEFAULT_EMAIL_BODY'
                        );

                        foreach ($formValidData as $name => $validData) {
                            $type      = $validData['data']['general']['type'];
                            $label     = $validData['label'];
                            $separator = ', ';

                            if (null === $validData['value']) {
                                $value = [];
                            } elseif ($type === 'Upload') {
                                $value = array_keys($uploadedFiles);
                            } elseif ($type === 'Calendar') {
                                $value = is_array($validData['value']) ? $validData['value'] : [$validData['value']];
                                foreach ($value as &$val) {
                                    try {
                                        $date = Factory::getDate($val, 'UTC');
                                        $date->setTimezone(new DateTimeZone($submissionData['userTimeZone']));
                                        $val = $date->format($validData['data']['options']['dateFormat'], true);
                                    } catch (Throwable $e) {
                                    }
                                }

                                if ($validData['data']['options']['mode'] === 'range') {
                                    $separator = ' -> ';
                                }
                            } else {
                                $value = is_array($validData['value']) ? $validData['value'] : [$validData['value']];
                            }

                            $value      = implode($separator, $value);
                            $clientBody = str_replace(
                                ['{' . $name . ':label}', '{' . $name . ':value}'],
                                [$label, $value],
                                $clientBody
                            );
                            $adminBody  = str_replace(
                                ['{' . $name . ':label}', '{' . $name . ':value}'],
                                [$label, $value],
                                $adminBody
                            );
                            $formData[] = '<strong>' . $label . '</strong>: ' . $value;
                        }

                        $formData  = implode('<br>', $formData);
                        $siteName  = $this->app->get('sitename');
                        $guestName = $user->name ?: '';

                        // Replace email subject
                        $searchSubject  = ['{website}', '{guestName}'];
                        $replaceSubject = [$siteName, $guestName];

                        // Replace email body
                        $searchBody  = ['{website}', '{guestName}', '{formData}'];
                        $replaceBody = [$siteName, $guestName, $formData];

                        // Check to send email to admin
                        if ($form->params->get('enableAdminEmail')) {
                            $adminEmails = trim($form->params->get('adminEmails', ''));

                            if (empty($adminEmails)) {
                                $adminEmails = trim($this->app->get('mailfrom', ''));
                            }

                            if (!empty($adminEmails)) {
                                $adminEmails = array_filter(explode(',', $adminEmails), function ($email) {
                                    return false !== filter_var($email, FILTER_VALIDATE_EMAIL);
                                });

                                if (!empty($adminEmails)) {
                                    $adminEmailSubject = (array)$form->params->get('adminEmailSubject', []);
                                    $adminSubject      = str_replace(
                                        $searchSubject,
                                        $replaceSubject,
                                        Text::_(
                                            trim($adminEmailSubject[$language] ?? $adminEmailSubject['en-GB'] ?? '') ?: 'COM_JLFORM_ADMIN_DEFAULT_EMAIL_SUBJECT'
                                        )
                                    );

                                    $adminBody = str_replace(
                                        $searchBody,
                                        $replaceBody,
                                        $adminBody
                                    );

                                    $submissionData['metadata']['emails']['admin'] = [
                                        'recipient' => $adminEmails,
                                        'subject'   => $adminSubject,
                                        'body'      => $adminBody,
                                    ];
                                }
                            }
                        }

                        // Check to send email to client
                        $clientEmails = [];

                        foreach ($formValidData as $validData) {
                            if ($validData['data']['general']['type'] === 'Text'
                                && $validData['data']['options']['format'] === 'email'
                                && !empty($validData['data']['options']['sendEmail'])
                                && false !== filter_var($validData['value'], FILTER_VALIDATE_EMAIL)
                            ) {
                                $clientEmails[] = $validData['value'];
                            }
                        }

                        $submissionData['metadata']['emails']['client']['recipient'] = $clientEmails;

                        if ($form->params->get('enableClientEmail') && $clientEmails) {
                            if (!empty($clientEmails)) {
                                $clientEmailSubject = (array)$form->params->get('clientEmailSubject', []);
                                $clientSubject      = str_replace(
                                    $searchSubject,
                                    $replaceSubject,
                                    Text::_(trim($clientEmailSubject[$language] ?? $clientEmailSubject['en-GB'] ?? '') ?: 'COM_JLFORM_CLIENT_DEFAULT_EMAIL_SUBJECT')
                                );

                                $submissionData['metadata']['emails']['client']['subject'] = $clientSubject;
                                $submissionData['metadata']['emails']['client']['body']    = str_replace($searchBody, $replaceBody, $clientBody);
                            }
                        }
                    }

                    $submissionTable->bind($submissionData);

                    if ($submissionTable->store()) {
                        $response['sid'] = (int)$submissionTable->id;
                        $attachments     = [];

                        if ($uploadedFiles) {
                            foreach ($uploadedFiles as $fileName => $uploadedFile) {
                                $dest = JPATH_ADMINISTRATOR . '/components/com_jlform/assets/files/' . $response['sid'] . '/' . $uploadedFile[0];

                                if (File::upload($uploadedFile[1], $dest) && !in_array($fileName, $ignoreAttachments)) {
                                    $attachments[$dest] = $fileName;
                                }
                            }
                        }

                        // Check to send emails
                        if (!empty($submissionData['metadata']['emails'])) {
                            $asyncHelper = new AsyncHelper();
                            foreach ($submissionData['metadata']['emails'] as $email) {
                                $asyncHelper->addTask(
                                    function () use ($email, $attachments) {
                                        if (!$email['recipient'] || !$email['subject'] || !$email['body']) {
                                            return;
                                        }

                                        /** @var Mail $mailer */
                                        $mailer = Factory::getContainer()->get(MailerFactoryInterface::class)->createMailer();
                                        $mailer->addRecipient($email['recipient'])
                                            ->isHtml()
                                            ->setSubject($email['subject'])
                                            ->setBody($email['body']);

                                        if ($attachments) {
                                            foreach ($attachments as $path => $name) {
                                                $mailer->addAttachment($path, $name);
                                            }
                                        }

                                        try {
                                            $mailer->send();
                                        } catch (Throwable $e) {
                                            if ($attachments) {
                                                // Remove the attachments and try again
                                                $mailer->clearAttachments();

                                                try {
                                                    $mailer->send();
                                                } catch (Throwable $e2) {
                                                }
                                            }
                                        }
                                    }
                                );
                            }

                            if ($attachments) {
                                ini_set('memory_limit', '-1');
                            }

                            $asyncHelper->run();
                        }

                        $afterSubmitEvent = new AfterSubmitEvent(['subject' => $form, 'submission' => $submissionTable]);
                        $app->getDispatcher()->dispatch(FormEvent::AFTER_SUBMIT, $afterSubmitEvent);

                        if (function_exists(FormEvent::AFTER_SUBMIT)) {
                            call_user_func_array(FormEvent::AFTER_SUBMIT, [$afterSubmitEvent]);
                        }

                        $response['eventBuffer'] = $afterSubmitEvent->getResult();

                        if ($redirect = $afterSubmitEvent->getRedirect()) {
                            $response['redirect'] = $redirect;
                        }

                        // Handle payment plugin
                        if ($paymentName && PluginHelper::isEnabled('jlform', $paymentName)) {
                            PluginHelper::importPlugin('jlform', $paymentName);
                            $amount             = (float)$form->plugins->get($paymentName . 'Amount', 0.00);
                            $paymentAmountEvent = new PaymentAmountEvent(
                                'onJLFormPaymentAmount',
                                [
                                    'subject'     => $form,
                                    'submission'  => $submissionTable,
                                    'amount'      => $amount,
                                    'currency'    => $form->plugins->get($paymentName . 'Currency', ''),
                                    'paymentName' => $paymentName,
                                ]
                            );
                            $app->getDispatcher()->dispatch($paymentAmountEvent->getName(), $paymentAmountEvent);
                            $submissionPaymentTable = $mvcFactory->createTable('SubmissionPayment', 'Administrator');
                            $submissionPaymentTable->bind([
                                'submissionId' => $submissionTable->id,
                                'formId'       => $submissionTable->formId,
                                'name'         => $paymentName,
                                'status'       => 'Pending',
                                'createdBy'    => $submissionTable->createdBy,
                                'createdDate'  => $submissionTable->createdDate,
                                'amount'       => $paymentAmountEvent->getAmount(),
                                'currency'     => $paymentAmountEvent->getCurrency(),
                                'data'         => [
                                    'userGroups' => $form->plugins->get($paymentName . 'UserGroups', []),
                                ],
                            ]);
                            $submissionPaymentTable->store();
                            $response['payment'] = [
                                'id'       => (int)$submissionPaymentTable->id,
                                'name'     => $submissionPaymentTable->name,
                                'currency' => $submissionPaymentTable->currency,
                                'amount'   => $paymentAmountEvent->getAmount(),
                            ];
                        }
                    } else {
                        $response['errors'] = [
                            '__system' => [
                                Text::_('COM_JLFORM_ERROR_CAN_NOT_SUBMIT_DATA_MSG'),
                                implode(', ', $submissionTable->getErrors()),
                            ],
                        ];
                    }
                }
            }

            $response['eventBuffer'] = implode(PHP_EOL, ArrayHelper::arrayUnique($response['eventBuffer']));

            echo new JsonResponse($response);
        } catch (Throwable $e) {
            echo new JsonResponse(['errors' => ['__system' => [$e->getMessage()]]], null, true);
        }
    }
}
