const handleFormStep = (formContainer, isNext) => {
    const currentMain = formContainer.querySelector('.jl-form__main-step[data-step].active');
    const currentStep = parseInt(currentMain.dataset.step);
    const targetStep = (currentStep + (isNext ? 1 : -1)).toString();
    const currentBar = formContainer.querySelector('.jl-form__step-bar__group[data-step="' + currentStep + '"]');
    // const targetBar = formContainer.querySelector('.jl-form__step-bar__group[data-step="' + targetStep + '"]');
    const targetMain = formContainer.querySelector('.jl-form__main-step[data-step="' + targetStep + '"]');

    currentMain?.classList.remove('active');
    targetMain?.classList.add('active');

    if (isNext) {
        currentBar.nextElementSibling.classList.add('active');
    } else {
        currentBar.classList.remove('active');
    }

    formContainer.querySelectorAll('[data-jl-calendar]').forEach(calendar => {
        if (typeof initJLCalendar === 'function') {
            initJLCalendar(calendar);
        }
    });

    document.dispatchEvent(new CustomEvent('onJLFormStepActive', {
        detail: { formContainer, mainContainer: currentMain, stepContainer: targetMain, step: currentStep }
    }));

    formContainer.scrollIntoView(0);
};

document.addEventListener('DOMContentLoaded', () => {
    // Switcher
    document.querySelectorAll('.jl-form__field-switcher').forEach((switcher) => {
        switcher.querySelector('input[type="checkbox"]')?.addEventListener('input', function () {
            const hint = switcher.querySelector('.jl-form__field-switcher__hint');

            if (hint) {
                hint.innerText = this.checked ? hint.dataset.lblYes : hint.dataset.lblNo;
            }
        });
    });

    // Label hint for List
    document.querySelectorAll('select[name^="JLForm["]:not([multiple]), input[type="radio"][name^="JLForm["]').forEach((input) => {
        const label = document.querySelector('label[for="' + (input.id ?? '__NONE_ID__') + '"]');

        if (label && label.querySelector('[data-lbl-hint]')) {
            const displayLblHint = () => {
                label.querySelectorAll('[data-lbl-hint]').forEach(hint => {
                    hint.hidden = true;
                    hint.innerHTML = '';
                });
                const value = input.nodeName === 'INPUT' ? (input.checked ? input.value : null) : input.value;

                if (value !== null) {
                    const hint = label.querySelector('[data-lbl-hint^="' + value + ':"]');

                    if (hint) {
                        hint.hidden = false;
                        hint.innerHTML = hint.dataset.lblHint.replace(value + ':', '').trim();
                    }
                }
            };

            input.addEventListener('change', displayLblHint);
            displayLblHint();
        }
    });

    // Payment toggle visible
    document.querySelectorAll('.jl-form__payments-list').forEach(paymentList => {
        paymentList.querySelectorAll('input[name="JLForm[__payment]"]').forEach(input => {
            input.addEventListener('input', () => {
                paymentList.querySelectorAll('[data-jlform-payment-visible]').forEach(paymentItemContainer => {
                    const value = paymentItemContainer.dataset.jlformPaymentVisible;
                    paymentItemContainer.hidden = paymentList.querySelector('input[name="JLForm[__payment]"]:checked')?.value !== value;
                });
            });
        });
    });


    // Validate/Submit data
    document.querySelectorAll('.jl-form__main-step__action--back, .jl-form__main-step__action--next, .jl-form__main-step__action--finish').forEach((btn) => {
        btn.addEventListener('click', async () => {
            const formContainer = btn.closest('.jl-form__container[data-form-id]');
            const stepContainer = btn.closest('[data-step]');

            if (!formContainer || !stepContainer) {
                return;
            }

            // Reset and hide all messages
            Joomla.removeMessages();
            stepContainer.querySelectorAll('.jl-form__field__message-feedback').forEach((msg) => {
                msg.innerHTML = '';
                msg.setAttribute('hidden', '');
            });

            if (btn.classList.contains('jl-form__main-step__action--back')) {
                handleFormStep(formContainer, false);
                return;
            }

            const isSubmit = btn.classList.contains('jl-form__main-step__action--finish');
            const step = isSubmit ? undefined : parseInt(stepContainer.dataset.step);
            const formId = parseInt(formContainer.dataset.formId);

            if ((!isSubmit && isNaN(step)) || isNaN(formId)) {
                return;
            }

            const formData = new FormData();
            const inputs = (isSubmit ? formContainer : stepContainer).querySelectorAll(
                'input[name^="JLForm["], select[name^="JLForm["], textarea[name^="JLForm["]',
            );

            for (const input of inputs) {
                const name = input.name;
                switch (input.nodeName.toUpperCase()) {
                    case 'INPUT':
                        if (input.type?.toLowerCase() === 'file') {
                            [...input.files].forEach((file) => formData.append(name, file));
                            continue;
                        }

                        if (!['checkbox', 'check', 'radio'].includes(input.type?.toLowerCase()) || input.checked) {
                            formData.append(name, input.value);
                        }

                        break;

                    case 'SELECT':
                        if (input.multiple) {
                            input
                                .querySelectorAll('option:checked')
                                .entries()
                                .forEach(([, opt]) => formData.append(name, opt.value));
                        } else {
                            formData.append(name, input.value);
                        }

                        break;

                    default:
                        formData.append(name, input.value);
                        break;
                }

                if (input.closest('.jl-form__field-container')?.hasAttribute('hidden')) {
                    formData.append('JLForm[__hiddenFields][]', name.replace(/(\[])$/g, '').replace(/^(JLForm\[)|]$/g, ''));
                }
            }

            formData.set('formId', formId.toString());

            if (step) {
                formData.set('step', (step - 1).toString());
            }

            const csrf = Joomla.getOptions('csrf.token');
            const root = Joomla.getOptions('system.paths').root;
            formData.set(csrf, '1');
            formData.set('rangeSeparator', window.flatpickr?.l10ns?.default?.rangeSeparator ?? ' to ');
            formData.set('userTimeZone', Intl?.DateTimeFormat()?.resolvedOptions()?.timeZone ?? '');
            btn.setAttribute('disabled', '');

            // Show loader
            const loader = Joomlab.loader.create(stepContainer).show();

            const closeRequest = () => {
                btn.removeAttribute('disabled');
                loader.close();
            }

            const {recaptchaSiteKey, recaptchaVersion} = Joomla.getOptions('com_jlform');
            let isValidCaptcha = true;

            if (isSubmit
                && recaptchaSiteKey
                && grecaptcha
            ) {
                if (recaptchaVersion === 3) {
                    await new Promise(resolve => {
                        grecaptcha.ready(() => {
                            grecaptcha.execute(recaptchaSiteKey, {action: 'submit'}).then((token) => {
                                formData.set('g-recaptcha-response', token);
                                resolve(token)
                            }).catch(err => {
                                isValidCaptcha = false;
                                resolve(isValidCaptcha);
                                console.debug('Recaptcha ERR:', err);
                            });
                        });
                    });
                } else {
                    isValidCaptcha = Joomla.getOptions('JLFormRecaptcha2IsValid', false);
                    formData.set('g-recaptcha-response', Joomla.getOptions('JLFormRecaptcha2Response'));
                }
            }

            if (!isValidCaptcha) {
                Joomlab.modal.error(Joomla.Text._('COM_JLFORM_INVALID_RECAPTCHA_MSG'));

                return closeRequest();
            }

            const paymentElement = formContainer.querySelector('input[name="JLForm[__payment]"]:checked');

            if (isSubmit) {
                const beforeSubmitEvent = new CustomEvent('onJLFormBeforeSubmit', {
                    detail: {
                        handlers: [],
                        errors: [],
                        formData,
                        paymentElement,
                        formContainer,
                    }
                });

                // Dispatch an event before submit
                document.dispatchEvent(beforeSubmitEvent);
                const {handlers, errors} = beforeSubmitEvent.detail;

                if (handlers.length) {
                    for (const handler of handlers) {
                        if (typeof handler === 'function') {
                            await handler();
                        }
                    }
                }

                if (errors.length) {
                    for (const {field, message} of errors) {
                        let error = message;

                        if (Array.isArray(error)) {
                            error = error.join('<br>');
                        } else if (error && typeof error === 'object') {
                            error = error.message || Object.entries(error).map(([k, v]) => `${k}: ${v}`).join('<br>');
                        } else {
                            error = error.toString();
                        }

                        let fieldElement = field;

                        if (typeof fieldElement === 'string') {
                            fieldElement = formContainer.querySelector(fieldElement);
                        }

                        const messageElement = fieldElement
                            ?.closest('.jl-form__field-container')
                            ?.querySelector('.jl-form__field__message-feedback');

                        if (messageElement) {
                            messageElement.innerHTML = error;
                            messageElement.removeAttribute('hidden');
                        } else {
                            Joomlab.modal.error(error);
                        }
                    }

                    return closeRequest();
                }
            }

            const finaliseContainer = formContainer.querySelector('.jl-form__submission-finalise');
            const finaliseBody = finaliseContainer.querySelector('.jl-form__submission-finalise-body');
            fetch(root + '/index.php?option=com_jlform&task=ajax.processFormData&format=json', {
                method: 'POST',
                body: formData,
            })
                .then((response) => response.json())
                .then(async (json) => {

                    const {errors} = json.data;

                    if (Object.keys(errors).length) {
                        // Validate failure
                        if (errors.__system || errors.__payment) {
                            if (errors.__system) {
                                Joomlab.modal.error(errors.__system);
                            }

                            if (errors.__payment) {
                                const pmMsgElement = stepContainer.querySelector('.jl-form__payments-message');

                                if (pmMsgElement) {
                                    pmMsgElement.innerHTML = errors.__payment.join('<br>');
                                    pmMsgElement.removeAttribute('hidden');
                                }
                            }
                        }

                        for (const name in errors) {
                            if (name === '__system' || name === '__payment') {
                                continue;
                            }

                            const messageElement = stepContainer.querySelector('[name^="JLForm[' + name + ']"]')
                                ?.closest('.jl-form__field-container')
                                ?.querySelector('.jl-form__field__message-feedback');

                            if (!messageElement) {
                                continue;
                            }

                            const errorArray = Array.isArray(errors[name]) ? errors[name] : [errors[name]];
                            messageElement.innerHTML = errorArray.join('<br>');
                            messageElement.removeAttribute('hidden');
                        }

                        setTimeout(() => formContainer.scrollIntoView({behavior: 'smooth'}), 350);
                    } else {
                        // Almost done
                        const redirect = (json.data.redirect || '').trim();

                        if (redirect) {
                            window.location.href = redirect;
                        } else {
                            if (isSubmit) {
                                const eventBuffer = json.data.eventBuffer.trim();
                                if (eventBuffer.length) {
                                    const htmlElement = document.createElement('div');
                                    htmlElement.classList.add('jl-form__submission-finalise-body__html');
                                    htmlElement.innerHTML = eventBuffer;
                                    htmlElement.querySelectorAll('script').forEach(js => {
                                        const script = document.createElement('script');

                                        if (js.src) {
                                            script.src = js.src;
                                        } else {
                                            script.textContent = js.textContent;
                                        }

                                        const pNode = js.parentNode;
                                        pNode.insertBefore(script, js);
                                        pNode.removeChild(js);
                                    });

                                    finaliseBody.appendChild(htmlElement);
                                }

                                const detail = {
                                    formContainer,
                                    paymentElement,
                                    finaliseBody,
                                    handlers: [],
                                };

                                const afterSubmitEvent = new CustomEvent('onJLFormAfterSubmit', {detail});

                                // Dispatch an event before submit
                                document.dispatchEvent(afterSubmitEvent);

                                if (paymentElement && json.data.payment) {
                                    detail.doPaymentUrl = Joomla.getOptions('com_jlform').paymentBaseUrl.replace('sid=', `sid=${json.data.sid}`);
                                    detail.payment = new JLFormPayment(finaliseBody, json.data.payment);

                                    // Dispatch payment event
                                    const paymentName = json.data.payment.name[0].toUpperCase() + json.data.payment.name.substring(1);
                                    document.dispatchEvent(new CustomEvent(`onJLFormPayment${paymentName}`, {detail}));
                                }

                                if (detail.handlers.length) {
                                    for (const handler of detail.handlers) {
                                        if (typeof handler === 'function') {
                                            await handler();
                                        }
                                    }
                                }

                                formContainer.querySelectorAll('.jl-form__step-bar, .jl-form__main').forEach(el => el.setAttribute('hidden', ''));
                                finaliseContainer.removeAttribute('hidden');
                                setTimeout(() => formContainer.scrollIntoView(true), 100);
                            } else {
                                handleFormStep(formContainer, true);
                            }
                        }
                    }
                })
                .finally(() => {
                    btn.removeAttribute('disabled');
                    loader.close();
                });
        });
    });

    // Terms and conditions
    document.querySelectorAll('[data-jlform-tnc] > a').forEach(a => {
        a.addEventListener('click', e => {
            e.preventDefault();
            e.stopPropagation();
            const tncText = (a.closest('[data-jlform-tnc]')?.dataset.jlformTnc || '').trim();

            if (tncText) {
                Joomlab.modal.dialog(tncText, {size: 'large'});
            }
        });
    });
});

function onJLFormRecaptcha2Callback(token) {
    const {recaptchaSiteKey, recaptchaVersion} = Joomla.getOptions('com_jlform');

    if (!window.grecaptcha
        || !recaptchaSiteKey
        || recaptchaVersion !== 2) {

    }

    Joomla.loadOptions({ JLFormRecaptcha2IsValid: true, JLFormRecaptcha2Response: token });

}

function onJLFormRecaptcha2ExpiredCallback() {
    if (window.grecaptcha) {
        Joomla.loadOptions({ JLFormRecaptcha2IsValid: false, JLFormRecaptcha2Response: '' });
        grecaptcha.reset();
    }
}
