<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_joomlab
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\Joomlab\Helper;

defined('_JEXEC') or die;

use Fiber;
use Throwable;

final class AsyncHelper
{
    private array $tasks = [];

    public function __construct(private readonly array $config = []) {}

    public function addTask(callable $task): static
    {
        $fiber = new Fiber($task);

        if (!empty($this->config['autoStart'])) {
            try {
                $fiber->start();
            } catch (Throwable $e) {
            }
        }

        $this->tasks[] = $fiber;

        return $this;
    }

    public function getTasks(): array
    {
        return $this->tasks;
    }

    public function run(): array
    {
        $results = [];

        if (!$this->count()) {
            return $results;
        }

        // Continue processing until all fibers are complete
        while (!empty($this->tasks)) {
            foreach ($this->tasks as $index => $fiber) {
                try {
                    /** @var Fiber $fiber */
                    if (!$fiber->isStarted()) {
                        $fiber->start();
                    } elseif ($fiber->isSuspended()) {
                        $fiber->resume();
                    }

                    if ($fiber->isTerminated()) {
                        unset($this->tasks[$index]);
                        $results[$index] = [
                            'success' => true,
                            'data'    => $fiber->getReturn(),
                        ];
                    }
                } catch (Throwable $e) {
                    if (!empty($this->config['strict'])) {
                        throw $e;
                    }

                    unset($this->tasks[$index]);
                    $results[$index] = [
                        'success' => false,
                        'data'    => $e,
                    ];
                }

                $this->tasks = array_values($this->tasks);
            }

            $usleep = $this->config['usleep'] ?? 100000; // Defaults to 100 milliseconds

            if (!in_array($usleep, [null, false, 0]) && is_numeric($usleep)) {
                usleep((int)$usleep);
            }
        }

        return $results;
    }

    public function count(): int
    {
        return count($this->tasks);
    }
}