<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLForm\Site\Helper\FieldValidator;

defined('_JEXEC') or die;

use Joomla\CMS\Language\Text;
use Joomla\Component\JLForm\Site\Helper\FieldValidator;
use Joomla\Plugin\System\Joomlab\Media\FileType;

class UploadField extends FieldValidator
{
    public function validate(&$value = null): bool
    {
        $value = $this->app->input->files->get('JLForm', [], 'RAW')[$this->name] ?? [];
        $multi = !!$this->options->get('multiple');

        if (empty($value) || ($multi && empty($value[0]))) {
            if ($this->required) {
                $this->addError(Text::_('COM_JLFORM_ERR_FIELD_REQUIRED_MSG'));

                return false;
            }

            return true;
        }

        if (!is_array($value)) {
            $this->addError(Text::_('COM_JLFORM_ERR_FIELD_INVALID_ARRAY_MSG'));

            return false;
        }

        if (!$multi) {
            $value = [$value];
        }

        $this->doValidate($value);
        $this->value = $value;

        return empty($this->errors);
    }

    protected function doValidate(&$value)
    {
        $fileTypes   = explode('|', trim($this->options->get('fileType', 'jpg|jpeg|png|svg|gif')));
        $fileMaxSize = (float)($this->options->get('fileMaxSize') ?? 0);

        if ($fileMaxSize < 0) {
            $fileMaxSize = 0;
        }

        $fileMaxSizeMB = $fileMaxSize;
        $fileMaxSize   *= 1e+6;

        foreach ($value as $file) {
            if ($fileMaxSize && $file['size'] > $fileMaxSize) {
                $this->addError(
                    Text::sprintf(
                        'COM_JLFORM_ERR_FIELD_UPLOAD_FILE_MAX_SIZE_MSG',
                        $fileMaxSizeMB,
                        $file['name'],
                        round($file['size'] / 1e+6, 1)
                    )
                );
            }

            $isValidFileType = false;

            foreach ($fileTypes as $fileType) {
                if (FileType::extensionToMimeType($fileType) === $file['type']) {
                    $isValidFileType = true;
                    break;
                }
            }

            if (!$isValidFileType) {
                $this->addError(
                    Text::sprintf(
                        'COM_JLFORM_ERR_FIELD_UPLOAD_FILE_TYPE_MSG',
                        FileType::mimeTypeToExtension($file['type'])
                    )
                );
            }
        }
    }
}
