<?php

/**
 * @package     Joomlab
 * @subpackage  com_jlform
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Component\JLForm\Site\Helper;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Helper\ModuleHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Component\JLForm\Site\Event\BeforeShowEvent;
use Joomla\Component\JLForm\Site\Event\FormEvent;
use Joomla\Component\JLForm\Site\Model\FormModel;
use Joomla\Plugin\System\Joomlab\Helper\AssetHelper;
use Joomla\Plugin\System\Joomlab\Helper\LayoutHelper;
use Joomla\Plugin\System\Joomlab\Helper\UtilityHelper;
use Joomla\Registry\Registry;
use stdClass;

class FormHelper
{
    public static function buildAttributes(array $attributes): string
    {
        $attrs = [];

        foreach ($attributes as $name => $value) {
            if (null === $value || (is_bool($value) && !$value)) {
                continue;
            }

            $attrs[] = $name . '="' . htmlspecialchars($value) . '"';
        }

        return implode(' ', $attrs);
    }

    public static function renderForm($formOrId): string
    {
        $form = is_numeric($formOrId) ? static::getForm($formOrId) : $formOrId;
        $app  = Factory::getApplication();

        if ($form->params->get('requireUserLogin') && $app->getIdentity()->guest) {
            $loginModule         = ModuleHelper::getModule('mod_login');
            $loginModule->id     = uniqid();
            $loginModule->params = new Registry($loginModule->params);
            $loginModule->params->set('pretext', Text::_('COM_JLFORM_FORM_REQUIRE_USER_LOGIN_TEXT'));

            return ModuleHelper::renderModule($loginModule);
        }

        $formLayout = static::buildFormLayout($form);

        if (is_string($formLayout)) {
            return $formLayout;
        }

        return LayoutHelper::render(
            'form.display',
            $formLayout,
            JPATH_SITE . '/components/com_jlform/layouts',
            ['component' => 'com_jlform', 'client' => 0]
        );
    }

    public static function getForm(int $id): stdClass
    {
        /** @var FormModel $formModel */
        $formModel = Factory::getApplication()
            ->bootComponent('com_jlform')
            ->getMVCFactory()
            ->createModel('Form', 'Site', ['ignore_request' => true]);

        return $formModel->getItem($id);
    }

    public static function _(array|stdClass $data, bool $defLang = false): string
    {
        $data        = (array)$data;
        $defaultLang = ComponentHelper::getParams('com_languages')->get('site', 'en-GB');
        $langTag     = $defLang ? $defaultLang : Factory::getApplication()->getLanguage()->getTag();

        return trim($data[$langTag] ?? $data[$defaultLang] ?? $data['en-GB'] ?? '');
    }

    public static function buildFormLayout($formOrId): array|string
    {
        PluginHelper::importPlugin('jlform');
        $buildSteps       = [];
        $app              = Factory::getApplication();
        $form             = is_numeric($formOrId) ? static::getForm($formOrId) : $formOrId;
        $formLayout       = $form->params->get('formLayout', 'bootstrap');
        $horizontal       = !!$form->params->get('formHorizontal', '0');
        $config           = ComponentHelper::getParams('com_jlform');
        $recaptchaVersion = (int)$config->get('useRecaptcha');
        $recaptchaSiteKey = trim($config->get('recaptchaSiteKey' . $recaptchaVersion, ''));
        $useRecaptcha     = $form->params->get('useRecaptcha', '0')
            && in_array($recaptchaVersion, [2, 3])
            && $recaptchaSiteKey;
        $dispatcher       = $app->getDispatcher();
        $basePath         = PluginHelper::isEnabled('jlform', 'pro') ? JPATH_PLUGINS . '/jlform/pro/tmpl' : '';
        $layoutOptions    = ['component' => 'com_jlform', 'client' => 0, 'basePathReverse' => true];
        $data             = $app->getInput()->get('JLForm', [], 'ARRAY');

        foreach ($form->data as $step) {
            if (empty($step['rows'])) {
                continue;
            }

            $stepTitle = static::_($step['title'] ?? []) ?: 'Main';

            if (!isset($buildSteps[$stepTitle])) {
                $buildSteps[$stepTitle] = [];
            }

            foreach ($step['rows'] as $i => $row) {
                if (empty($row)) {
                    continue;
                }

                $rowDisplayData = ['index' => $i, 'options' => []];

                foreach ($row as $column) {
                    $columnSize      = (int)$column['size'];
                    $rowDisplayDatum = ['size' => $columnSize, 'fields' => []];

                    foreach ($column['data'] as $field) {
                        $type       = $field['general']['type'] ?? '';
                        $options    = new Registry($field['options'] ?? []);
                        $isFreeText = $type === 'Text' && $options->get('format') === 'freeText';

                        if ($type && ($isFreeText || !empty($field['general']['name']))) {
                            $required    = (bool)($field['general']['required'] ?? false);
                            $name        = trim($field['general']['name'] ?? '');
                            $showOn      = trim($field['general']['showOn'] ?? '');
                            $displayData = [
                                'name'       => $isFreeText ? '' : 'JLForm[' . $name . ']',
                                'label'      => $isFreeText ? '' : static::_($field['general']['label']),
                                'hint'       => $isFreeText ? '' : static::_($field['general']['hint']),
                                'message'    => $isFreeText ? '' : static::_($field['general']['message']),
                                'options'    => $options,
                                'formParams' => $form->params,
                                'horizontal' => !$isFreeText && ($field['general']['inlineLabel'] || $horizontal),
                                'hideLabel'  => !$isFreeText && ($field['general']['hideLabel'] ?? false),
                                'required'   => !$isFreeText && $required,
                                'value'      => $data[$name] ?? null,
                                'lblAttrs'   => [],
                                'ctrlAttrs'  => [],
                            ];

                            if ($displayData['options']->get('multiple') || $type === 'CheckList') {
                                $displayData['name'] .= '[]';
                            }

                            if ($isFreeText) {
                                $displayData['options']->set('freeText', static::_((array)$displayData['options']->get('freeText')));
                            }

                            if (in_array($type, ['DropdownList', 'RadioList', 'CheckList'])) {
                                $displayData['options']->set(
                                    'items',
                                    array_map(
                                        function ($item) use (&$displayData) {
                                            $optionDefault = static::_($item, true);
                                            $optionNative  = static::_($item);
                                            $jsonDefault   = UtilityHelper::parseJson($optionDefault);
                                            $jsonNative    = UtilityHelper::parseJson($optionNative);
                                            $value         = trim($jsonNative['val'] ?? $jsonDefault['val'] ?? $optionNative);
                                            $text          = trim($jsonNative['txt'] ?? $jsonDefault['txt'] ?? $value);
                                            $isDefault     = (bool)($jsonNative['defVal'] ?? $jsonDefault['defVal'] ?? false);

                                            if ($lbl = trim($jsonNative['lbl'] ?? $jsonDefault['lbl'] ?? '')) {
                                                $displayData['label'] .= '<span class="jl-form__lbl-hint" data-lbl-hint="' . htmlspecialchars($value . ':' . $lbl) . '" hidden></span>';
                                            }

                                            return ['text' => $text, 'value' => $value, 'isDefault' => $isDefault];
                                        },
                                        $displayData['options']->get('items', [])
                                    )
                                );
                            }

                            if ($lblAttrs = static::parseAttributes($field['general']['lblAttrs'] ?? '')) {
                                $displayData['lblAttrs'] = $lblAttrs;
                            }

                            if ($ctrlAttrs = static::parseAttributes($field['general']['ctrlAttrs'] ?? '')) {
                                unset($ctrlAttrs['name'], $ctrlAttrs['label'], $ctrlAttrs['data-show-on']);
                                $displayData['ctrlAttrs'] = $ctrlAttrs;
                            }

                            if ($showOn) {
                                AssetHelper::getWebAssetManager()->useScript('com_jlform.showon');
                                $displayData['ctrlAttrs']['data-show-on'] = json_encode(\Joomla\CMS\Form\FormHelper::parseShowOnConditions($showOn));
                            }

                            $displayData['ctrlAttrs']['class'] = trim(
                                $displayData['ctrlAttrs']['class'] . ' jl-form__field--' . $type
                            );

                            if ($required) {
                                $displayData['ctrlAttrs']['required'] = '';
                            }

                            if (empty($displayData['ctrlAttrs']['id'])) {
                                $displayData['ctrlAttrs']['id'] = 'jl-form-field-' . $type . '-' . uniqid();
                            }

                            $displayData['lblAttrs']['for'] = $displayData['ctrlAttrs']['id'];
                            $rowDisplayDatum['fields'][]    = LayoutHelper::render(
                                'form.field-' . $formLayout,
                                array_merge(
                                    $displayData,
                                    [
                                        'input' => LayoutHelper::render(
                                            'form.field.' . $type,
                                            $displayData,
                                            $basePath,
                                            $layoutOptions
                                        ),
                                    ]
                                ),
                                $basePath,
                                $layoutOptions
                            );
                        }
                    }

                    $rowDisplayData['options'][] = $rowDisplayDatum;
                }

                $buildSteps[$stepTitle][] = LayoutHelper::render(
                    'form.row-' . $formLayout,
                    $rowDisplayData,
                    $basePath,
                    $layoutOptions
                );
            }
        }

        // Event buffers
        $eventBuffers    = [];
        $beforeShowEvent = new BeforeShowEvent(['subject' => $form, 'buildSteps' => $buildSteps]);
        $dispatcher->dispatch(FormEvent::BEFORE_SHOW, $beforeShowEvent);
        $customPhpFile = JPATH_ADMINISTRATOR . '/components/com_jlform/assets/customs/form-' . $form->id . '.php';

        if (is_file($customPhpFile)) {
            include_once $customPhpFile;

            if (function_exists(FormEvent::BEFORE_SHOW)) {
                call_user_func_array(FormEvent::BEFORE_SHOW, [$beforeShowEvent]);
            }
        }

        $customBuffer = trim(implode(PHP_EOL, $beforeShowEvent->getResult()));

        if (!empty($customBuffer)) {
            return $customBuffer;
        }

        $buildSteps = $beforeShowEvent->getBuildSteps();

        // Payments list
        $paymentsList = [];

        foreach ($form->plugins->toArray() as $key => $value) {
            if (str_starts_with($key, 'use') && $value) {
                $plugin = strtolower(substr($key, 3));

                if ($form->plugins->get($plugin . 'PaymentPlugin') && PluginHelper::isEnabled('jlform', $plugin)) {
                    $layoutBuffer = LayoutHelper::render(
                        'form.payment.' . $plugin,
                        ['form' => $form],
                        JPATH_PLUGINS . '/jlform/' . $plugin . '/tmpl',
                        $layoutOptions
                    );

                    if (!empty($layoutBuffer)) {
                        $paymentsList[] = [
                            'name'    => $plugin,
                            'display' => $layoutBuffer,
                        ];
                    }
                }
            }
        }

        if (!empty($paymentsList)) {
            $paymentsListInput = LayoutHelper::render(
                'form.payment',
                [
                    'form'         => $form,
                    'paymentsList' => $paymentsList,
                ],
                $basePath,
                $layoutOptions
            );

            $eventBuffers[] = LayoutHelper::render(
                'form.field-' . $formLayout,
                [
                    'form'       => $form,
                    'horizontal' => $horizontal,
                    'required'   => true,
                    'hideLabel'  => false,
                    'lblAttrs'   => ['for' => ''],
                    'label'      => Text::_('COM_JLFORM_PAYMENT_LABEL'),
                    'input'      => $paymentsListInput,
                ],
                $basePath,
                $layoutOptions
            );
        }

        if ($useRecaptcha) {
            Text::script('COM_JLFORM_INVALID_RECAPTCHA_MSG');
            $app->getDocument()->addScriptOptions('com_jlform', [
                'recaptchaSiteKey' => $recaptchaSiteKey,
                'recaptchaVersion' => $recaptchaVersion,
            ]);

            $uri = 'https://www.google.com/recaptcha/api.js' . ($recaptchaVersion === 3 ? '?render=' . $recaptchaSiteKey : '');

            if ($recaptchaVersion === 2) {
                $eventBuffers[] = '<div class="g-recaptcha" data-sitekey="' . $recaptchaSiteKey . '" data-callback="onJLFormRecaptcha2Callback" data-expired-callback="onJLFormRecaptcha2ExpiredCallback"></div>';
            }

            AssetHelper::getWebAssetManager()->registerAndUseScript('com_jlform.recaptcha', $uri, [], ['defer' => true]);
        }

        return [
            'form'             => $form,
            'formBuffers'      => $buildSteps,
            'eventBuffers'     => $eventBuffers,
            'finishButtonText' => trim(static::_($form->params->get('finishButtonText', []))) ?: Text::_('COM_JLFORM_STEP_FINISH'),
        ];
    }

    public static function parseAttributes(string $attribute): array
    {
        $attributes  = [];
        $attribute   = trim($attribute);
        $ignoreAttrs = ['name', 'value', 'required'];

        if ($attribute) {
            $regex = '/[a-z0-9_-]+\s*=\s*["\'][^\'"]*["\']/i';

            while (preg_match($regex, $attribute, $matches)) {
                $attribute = trim(preg_replace($regex, '', $attribute, 1));
                [$name, $value] = preg_split('/\s*=\s*/', $matches[0], 2);

                if (
                    !in_array($name, $ignoreAttrs)
                    && preg_match('/^[a-z0-9-_]+$/i', trim($name))
                    && preg_match('/^["\']([^\'"]*)["\']$/i', trim($value))
                ) {
                    $attributes[trim($name)] = str_replace(['\'', '"'], '', trim($value));
                }
            }
        }

        if (!isset($attributes['class'])) {
            $attributes['class'] = '';
        }

        if (!isset($attributes['id'])) {
            $attributes['id'] = '';
        }

        return array_unique($attributes);
    }
}
