<?php

/**
 * @package     Joomlab
 * @subpackage  plg_system_joomlab
 * @copyright   Copyright (C) 2025 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\Joomlab\Helper;

use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\DatabaseInterface;
use Joomla\Event\Event;

defined('_JEXEC') or die;

class DashboardHelper
{
    private array $items = [];

    private readonly string $component;

    public function __construct(?string $component = null)
    {
        $this->component = $component ?? Factory::getApplication()->getInput()->get('option');
        Factory::getApplication()
            ->getLanguage()
            ->load($this->component . '.sys', JPATH_ADMINISTRATOR . '/components/' . $this->component);
    }

    public function addItem(array ...$items): static
    {
        $formToken = Session::getFormToken();

        foreach ($items as &$item) {
            if (isset($item['url'])
                && Uri::isInternal($item['url'])
                && !str_contains($item['url'], $formToken)
            ) {
                $item['url'] .= (str_contains($item['url'], '?') ? '&' : '?') . $formToken . '=1';
            }
        }

        array_push($this->items, ...$items);

        return $this;
    }

    public function getItems(): array
    {
        HTMLHelper::_('joomlab.framework', true);
        $app = Factory::getApplication();
        $app->getDispatcher()
            ->dispatch('onJLItemsDashboard', new Event('onJLItemsDashboard', ['subject' => $this]));
        $user  = $app->getIdentity();
        $items = $this->items;

        if ($user->authorise('core.admin', $this->component) || $user->authorise('core.options', $this->component)) {
            $items[] = [
                'text' => Text::_(strtoupper($this->component) . '_MENU_CONFIGURATION'),
                'icon' => 'settings',
                'url'  => Route::_('index.php?option=com_config&view=component&component=' . $this->component . '&path=&return=' . base64_encode(Uri::getInstance()->toString()),
                    false),
                'card' => 'secondary',
            ];
        }

        return $items;
    }

    public function renderDownloadKeyInput(string $ext): string
    {
        $ajaxUrl = Route::_('index.php?option=com_ajax&plugin=joomlab&group=system&format=json', false);
        AssetHelper::getWebAssetManager()
            ->addInlineScript(<<<JAVASCRIPT
document.addEventListener('DOMContentLoaded', function() {
    const input = document.getElementById('jl-update-key-input');
    input?.addEventListener('keyup', function(e) {
        if (e.key.toUpperCase() === 'ENTER') {
            Joomlab.request.post('$ajaxUrl', { uKey: input.value.trim(), ext: '$ext' });
        }
    });
    
    document.getElementById('jl-update-key-button')?.addEventListener('click', function(e) {
        e.preventDefault();
        Joomlab.request.post('$ajaxUrl', { uKey: input.value.trim(), ext: '$ext' });
    });
    
    
});
JAVASCRIPT
            );
        $label = Text::_('PLG_SYSTEM_JOOMLAB_DOWNLOAD_KEY_LABEL');
        $desc  = Text::_('PLG_SYSTEM_JOOMLAB_DOWNLOAD_KEY_DESC');
        $hint  = htmlspecialchars(Text::_('PLG_SYSTEM_JOOMLAB_DOWNLOAD_KEY_HINT'));
        $uKey  = '';
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true)
            ->select($db->quoteName('extra_query'))
            ->from($db->quoteName('#__update_sites'))
            ->where($db->quoteName('name') . ' = :extension')
            ->bind(':extension', $ext);

        if ($extraQuery = $db->setQuery($query)->loadResult()) {
            $uKey = str_replace('uKey=', '', $extraQuery);
        }

        return <<<HTML
<div class="uk-margin">
    <label class="uk-form-label" for="jl-update-key-input">$label</label>
    <div class="uk-inline uk-width-large">
        <button class="uk-form-icon uk-form-icon-flip" id="jl-update-key-button" type="button" uk-icon="check"></button>
        <input class="uk-input" type="password" id="jl-update-key-input" placeholder="$hint" value="$uKey">
    </div>
    <p class="uk-text-meta">$desc</p>
</div>
HTML;
    }
}